unit vsop;

(* $define meeus *)  { Only use the accuracy as in the Meeus book }

interface

uses ah_math, sysutils;

type
  TVSOPEntry=record
    A,B,C: extended;
    end;
  TVSOPCalcFunc = function (nr,index: integer):TVSOPEntry of object;
  TVSOP=class(TObject)
  protected
    FDate: TDateTime;
    function LongitudeFactor(nr,index: integer):TVSOPEntry;  VIRTUAL; abstract;
    function LatitudeFactor(nr,index: integer):TVSOPEntry;  VIRTUAL; abstract;
    function RadiusFactor(nr,index: integer):TVSOPEntry;  VIRTUAL; abstract;
    function CalcLongitude:extended;
    function CalcLatitude:extended;
    function CalcRadius:extended;
    function Calc(factor: TVSOPCalcFunc):extended;
    procedure SetDate(value: TDateTime);
    function Tau:extended;
  public
    procedure DynamicToFK5(var longitude,latitude: extended);
    property Longitude:extended read CalcLongitude;
    property Latitude:extended read CalcLatitude;
    property Radius:extended read CalcRadius;
    property Date:TDateTime write SetDate;
    end;
  TCVSOP=class of TVSOP;
  TVSOPEarth=class(TVSOP)
  protected
    function LongitudeFactor(nr,index: integer):TVSOPEntry;  override;
    function LatitudeFactor(nr,index: integer):TVSOPEntry;  override;
    function RadiusFactor(nr,index: integer):TVSOPEntry;  override;
    end;

procedure earth_coord(date:TdateTime; var l,b,r: extended);

implementation
uses moon;

const datetime_2000_01_01: extended = 0;

procedure calc_coord(date: TDateTime; obj_class: TCVSOP; var l,b,r: extended);
var
  obj: TVSOP;
begin
  obj:=NIL;
  try
    obj:=obj_class.Create;
    obj.date:=date;
    r:=obj.radius;
    l:=obj.longitude;
    b:=obj.latitude;
    obj.DynamicToFK5(l,b);
  finally
    obj.free;
    end;
  l:=put_in_360(rad2deg(l));
  b:=rad2deg(b);
  end;

procedure earth_coord(date:TdateTime; var l,b,r: extended);
begin
  calc_coord(date,TVSOPEarth,l,b,r);
  end;

function TVSOP.CalcLongitude:extended;
begin
  result:=calc(Longitudefactor);
  end;
function TVSOP.CalcLatitude:extended;
begin
  result:=calc(Latitudefactor);
  end;
function TVSOP.CalcRadius:extended;
begin
  result:=calc(radiusfactor);
  end;
procedure TVSOP.SetDate(value: TDateTime);
begin
  FDate:=value;
  end;
function TVSOP.Tau:extended;
begin
  result:=(FDate-datetime_2000_01_01-0.5)/365250.0;
  end;

function TVSOP.Calc(factor: TVSOPCalcFunc):extended;
var
  t: extended;
  current: extended;
  r: array[0..5] of extended;
  i,j: integer;
begin
  t:=Tau;
  for j:=0 to 5 do begin
    r[j]:=0;
    i:=0;
    repeat
      WITH Factor(i,j) do
        current:=a*cos(b+c*t);
      r[j]:=r[j]+current;
      inc(i);
    until current=0;
    end;
  result:=(r[0]+t*(r[1]+t*(r[2]+t*(r[3]+t*(r[4]+t*r[5])))))*1e-8;
  end;

procedure TVSOP.DynamicToFK5(var longitude,latitude: extended);
var
  lprime,t: extended;
  delta_l, delta_b: extended;
begin
  t:=10*tau;
  lprime:=longitude+deg2rad(-1.397-0.00031*t)*t;
  delta_l:=-deg2rad(0.09033/3600)+deg2rad(0.03916/3600)*(cos(lprime)+sin(lprime))*tan(latitude);
  delta_b:=deg2rad(0.03916/3600)*(cos(lprime)-sin(lprime));
  longitude:=longitude+delta_l;
  latitude:=latitude+delta_b;
  end;

function TVSOPEarth.RadiusFactor(nr,index: integer):TVSOPEntry;
const
  (*$ifdef meeus *)
  vsop87_ear_r0:array[0.. 39,0..2] of extended = (
  (*$else *)
  vsop87_ear_r0:array[0..525,0..2] of extended = (
  (*$endif *)
  { 4330    1 }  ( 100013988.799, 0.00000000000,      0.00000000000 ),
  { 4330    2 }  (   1670699.626, 3.09846350771,   6283.07584999140 ),
  { 4330    3 }  (     13956.023, 3.05524609620,  12566.15169998280 ),
  { 4330    4 }  (      3083.720, 5.19846674381,  77713.77146812050 ),
  { 4330    5 }  (      1628.461, 1.17387749012,   5753.38488489680 ),
  { 4330    6 }  (      1575.568, 2.84685245825,   7860.41939243920 ),
  { 4330    7 }  (       924.799, 5.45292234084,  11506.76976979360 ),
  { 4330    8 }  (       542.444, 4.56409149777,   3930.20969621960 ),
  { 4330    9 }  (       472.110, 3.66100022149,   5884.92684658320 ),
  { 4330   10 }  (       328.780, 5.89983646482,   5223.69391980220 ),
  { 4330   11 }  (       345.983, 0.96368617687,   5507.55323866740 ),
  { 4330   12 }  (       306.784, 0.29867139512,   5573.14280143310 ),
  { 4330   13 }  (       174.844, 3.01193636534,  18849.22754997420 ),
  { 4330   14 }  (       243.189, 4.27349536153,  11790.62908865880 ),
  { 4330   15 }  (       211.829, 5.84714540314,   1577.34354244780 ),
  { 4330   16 }  (       185.752, 5.02194447178,  10977.07880469900 ),
  { 4330   17 }  (       109.835, 5.05510636285,   5486.77784317500 ),
  { 4330   18 }  (        98.316, 0.88681311277,   6069.77675455340 ),
  { 4330   19 }  (        86.499, 5.68959778254,  15720.83878487840 ),
  { 4330   20 }  (        85.825, 1.27083733351, 161000.68573767410 ),
  { 4330   21 }  (        62.916, 0.92177108832,    529.69096509460 ),
  { 4330   22 }  (        57.056, 2.01374292014,  83996.84731811189 ),
  { 4330   23 }  (        64.903, 0.27250613787,  17260.15465469040 ),
  { 4330   24 }  (        49.384, 3.24501240359,   2544.31441988340 ),
  { 4330   25 }  (        55.736, 5.24159798933,  71430.69561812909 ),
  { 4330   26 }  (        42.515, 6.01110242003,   6275.96230299060 ),
  { 4330   27 }  (        46.963, 2.57805070386,    775.52261132400 ),
  { 4330   28 }  (        38.968, 5.36071738169,   4694.00295470760 ),
  { 4330   29 }  (        44.661, 5.53715807302,   9437.76293488700 ),
  { 4330   30 }  (        35.660, 1.67468058995,  12036.46073488820 ),
  { 4330   31 }  (        31.921, 0.18368229781,   5088.62883976680 ),
  { 4330   32 }  (        31.846, 1.77775642085,    398.14900340820 ),
  { 4330   33 }  (        33.193, 0.24370300098,   7084.89678111520 ),
  { 4330   34 }  (        38.245, 2.39255343974,   8827.39026987480 ),
  { 4330   35 }  (        28.464, 1.21344868176,   6286.59896834040 ),
  { 4330   36 }  (        37.490, 0.82952922332,  19651.04848109800 ),
  { 4330   37 }  (        36.957, 4.90107591914,  12139.55350910680 ),
  { 4330   38 }  (        34.537, 1.84270693282,   2942.46342329160 ),
  { 4330   39 }  (        26.275, 4.58896850401,  10447.38783960440 ),
  (*$ifndef meeus *)
  { 4330   40 }  (        24.596, 3.78660875483,   8429.24126646660 ),
  { 4330   41 }  (        23.587, 0.26866117066,    796.29800681640 ),
  (*$endif *)
  { 4330   42 }  (        27.793, 1.89934330904,   6279.55273164240 )
  (*$ifndef meeus *)                                                 ,
  { 4330   43 }  (        23.927, 4.99598548138,   5856.47765911540 ),
  { 4330   44 }  (        20.349, 4.65267995431,   2146.16541647520 ),
  { 4330   45 }  (        23.287, 2.80783650928,  14143.49524243060 ),
  { 4330   46 }  (        22.103, 1.95004702988,   3154.68708489560 ),
  { 4330   47 }  (        19.506, 5.38227371393,   2352.86615377180 ),
  { 4330   48 }  (        17.958, 0.19871379385,   6812.76681508600 ),
  { 4330   49 }  (        17.174, 4.43315560735,  10213.28554621100 ),
  { 4330   50 }  (        16.190, 5.23160507859,  17789.84561978500 ),
  { 4330   51 }  (        17.314, 6.15200787916,  16730.46368959580 ),
  { 4330   52 }  (        13.814, 5.18962074032,   8031.09226305840 ),
  { 4330   53 }  (        18.833, 0.67306674027, 149854.40013480789 ),
  { 4330   54 }  (        18.331, 2.25348733734,  23581.25817731760 ),
  { 4330   55 }  (        13.641, 3.68516118804,   4705.73230754360 ),
  { 4330   56 }  (        13.139, 0.65289581324,  13367.97263110660 ),
  { 4330   57 }  (        10.414, 4.33285688538,  11769.85369316640 ),
  { 4330   58 }  (         9.978, 4.20126336355,   6309.37416979120 ),
  { 4330   59 }  (        10.169, 1.59390681369,   4690.47983635860 ),
  { 4330   60 }  (         7.564, 2.62560597390,   6256.77753019160 ),
  { 4330   61 }  (         9.661, 3.67586791220,  27511.46787353720 ),
  { 4330   62 }  (         6.743, 0.56270332741,   3340.61242669980 ),
  { 4330   63 }  (         8.743, 6.06359123461,   1748.01641306700 ),
  { 4330   64 }  (         7.786, 3.67371235637,  12168.00269657460 ),
  { 4330   65 }  (         6.633, 5.66149277792,  11371.70468975820 ),
  { 4330   66 }  (         7.712, 0.31242577789,   7632.94325965020 ),
  { 4330   67 }  (         6.592, 3.13576266188,    801.82093112380 ),
  { 4330   68 }  (         7.460, 5.64757188143,  11926.25441366880 ),
  { 4330   69 }  (         6.933, 2.92384586400,   6681.22485339960 ),
  { 4330   70 }  (         6.802, 1.42329806420,  23013.53953958720 ),
  { 4330   71 }  (         6.115, 5.13393615454,   1194.44701022460 ),
  { 4330   72 }  (         6.477, 2.64986648492,  19804.82729158280 ),
  { 4330   73 }  (         5.233, 4.62434053374,   6438.49624942560 ),
  { 4330   74 }  (         6.147, 3.02863936662, 233141.31440436149 ),
  { 4330   75 }  (         4.608, 1.72194702724,   7234.79425624200 ),
  { 4330   76 }  (         4.221, 1.55697533729,   7238.67559160000 ),
  { 4330   77 }  (         5.314, 2.40716580847,  11499.65622279280 ),
  { 4330   78 }  (         5.128, 5.32398965690,  11513.88331679440 ),
  { 4330   79 }  (         4.770, 0.25554312006,  11856.21865142450 ),
  { 4330   80 }  (         5.519, 2.09089154502,  17298.18232732620 ),
  { 4330   81 }  (         5.625, 4.34052903053,  90955.55169449610 ),
  { 4330   82 }  (         4.578, 4.46569641570,   5746.27133789600 ),
  { 4330   83 }  (         3.788, 4.90729383510,   4164.31198961300 ),
  { 4330   84 }  (         5.337, 5.09957905104,  31441.67756975680 ),
  { 4330   85 }  (         3.967, 1.20054555174,   1349.86740965880 ),
  { 4330   86 }  (         4.008, 3.03007204392,   1059.38193018920 ),
  { 4330   87 }  (         3.476, 0.76080277030,  10973.55568635000 ),
  { 4330   88 }  (         4.232, 1.05485713117,   5760.49843189760 ),
  { 4330   89 }  (         4.582, 3.76570026763,   6386.16862421000 ),
  { 4330   90 }  (         3.335, 3.13829943354,   6836.64525283380 ),
  { 4330   91 }  (         3.418, 3.00072390334,   4292.33083295040 ),
  { 4330   92 }  (         3.598, 5.70718084323,   5643.17856367740 ),
  { 4330   93 }  (         3.237, 4.16448773994,   9917.69687450980 ),
  { 4330   94 }  (         4.154, 2.59941292162,   7058.59846131540 ),
  { 4330   95 }  (         3.362, 4.54577697964,   4732.03062734340 ),
  { 4330   96 }  (         2.978, 1.30561268820,   6283.14316029419 ),
  { 4330   97 }  (         2.765, 0.51311975679,     26.29831979980 ),
  { 4330   98 }  (         2.802, 5.66263240521,   8635.94200376320 ),
  { 4330   99 }  (         2.927, 5.73787481548,  16200.77272450120 ),
  { 4330  100 }  (         3.164, 1.69140262657,  11015.10647733480 ),
  { 4330  101 }  (         2.598, 2.96244118586,  25132.30339996560 ),
  { 4330  102 }  (         3.519, 3.62639325753, 244287.60000722769 ),
  { 4330  103 }  (         2.676, 4.20725700850,  18073.70493865020 ),
  { 4330  104 }  (         2.978, 1.74971565805,   6283.00853968860 ),
  { 4330  105 }  (         2.287, 1.06975704977,  14314.16811304980 ),
  { 4330  106 }  (         2.863, 5.92838131397,  14712.31711645800 ),
  { 4330  107 }  (         3.071, 0.23793217002,  35371.88726597640 ),
  { 4330  108 }  (         2.656, 0.89959301780,  12352.85260454480 ),
  { 4330  109 }  (         2.415, 2.79975176257,    709.93304855830 ),
  { 4330  110 }  (         2.814, 3.51488206882,  21228.39202354580 ),
  { 4330  111 }  (         1.977, 2.61358297550,    951.71840625060 ),
  { 4330  112 }  (         2.548, 2.47684686575,   6208.29425142410 ),
  { 4330  113 }  (         1.999, 0.56090388160,   7079.37385680780 ),
  { 4330  114 }  (         2.305, 1.05376461628,  22483.84857449259 ),
  { 4330  115 }  (         1.855, 2.86090681163,   5216.58037280140 ),
  { 4330  116 }  (         2.157, 1.31396741861, 154717.60988768269 ),
  { 4330  117 }  (         1.970, 4.36929875289, 167283.76158766549 ),
  { 4330  118 }  (         1.635, 5.85571606764,  10984.19235169980 ),
  { 4330  119 }  (         1.754, 2.14452408833,   6290.18939699220 ),
  { 4330  120 }  (         2.154, 6.03828341543,  10873.98603048040 ),
  { 4330  121 }  (         1.714, 3.70157691113,   1592.59601363280 ),
  { 4330  122 }  (         1.541, 6.21598380732,  23543.23050468179 ),
  { 4330  123 }  (         1.611, 1.99824499377,  10969.96525769820 ),
  { 4330  124 }  (         1.712, 1.34295663542,   3128.38876509580 ),
  { 4330  125 }  (         1.642, 5.55026665339,   6496.37494542940 ),
  { 4330  126 }  (         1.502, 5.43948825854,    155.42039943420 ),
  { 4330  127 }  (         1.827, 5.91227480261,   3738.76143010800 ),
  { 4330  128 }  (         1.726, 2.16764983583,  10575.40668294180 ),
  { 4330  129 }  (         1.532, 5.35683107070,  13521.75144159140 ),
  { 4330  130 }  (         1.829, 1.66006148731,  39302.09696219600 ),
  { 4330  131 }  (         1.605, 1.90928637633,   6133.51265285680 ),
  { 4330  132 }  (         1.282, 2.46014880418,  13916.01910964160 ),
  { 4330  133 }  (         1.211, 4.41360631550,   3894.18182954220 ),
  { 4330  134 }  (         1.394, 1.77801929354,   9225.53927328300 ),
  { 4330  135 }  (         1.571, 4.95512957592,  25158.60171976540 ),
  { 4330  136 }  (         1.205, 1.19212540615,      3.52311834900 ),
  { 4330  137 }  (         1.132, 2.69830084955,   6040.34724601740 ),
  { 4330  138 }  (         1.504, 5.77002730341,  18209.33026366019 ),
  { 4330  139 }  (         1.393, 1.62621805428,   5120.60114558360 ),
  { 4330  140 }  (         1.077, 2.93931554233,  17256.63153634140 ),
  { 4330  141 }  (         1.232, 0.71655165307, 143571.32428481648 ),
  { 4330  142 }  (         1.087, 0.99769687939,    955.59974160860 ),
  { 4330  143 }  (         1.068, 5.28472576231,  65147.61976813770 ),
  { 4330  144 }  (         0.980, 5.10949204607,   6172.86952877200 ),
  { 4330  145 }  (         1.169, 3.11664290862,  14945.31617355440 ),
  { 4330  146 }  (         1.202, 4.02992510402,    553.56940284240 ),
  { 4330  147 }  (         0.979, 2.00000879212,  15110.46611986620 ),
  { 4330  148 }  (         0.962, 4.02380771400,   6282.09552892320 ),
  { 4330  149 }  (         0.999, 3.62643002790,   6262.30045449900 ),
  { 4330  150 }  (         1.030, 5.84989900289,    213.29909543800 ),
  { 4330  151 }  (         1.014, 2.84221578218,   8662.24032356300 ),
  { 4330  152 }  (         1.185, 1.51330541132,  17654.78053974960 ),
  { 4330  153 }  (         0.967, 2.67081017562,   5650.29211067820 ),
  { 4330  154 }  (         1.222, 2.65423784904,  88860.05707098669 ),
  { 4330  155 }  (         0.981, 2.36370360283,   6206.80977871580 ),
  { 4330  156 }  (         1.033, 0.13874927606,  11712.95531823080 ),
  { 4330  157 }  (         1.103, 3.08477302937,  43232.30665841560 ),
  { 4330  158 }  (         0.781, 2.53372735932,  16496.36139620240 ),
  { 4330  159 }  (         1.019, 3.04569392376,   6037.24420376200 ),
  { 4330  160 }  (         0.795, 5.80662989111,   5230.80746680300 ),
  { 4330  161 }  (         0.813, 3.57710279439,  10177.25767953360 ),
  { 4330  162 }  (         0.962, 5.31470594766,   6284.05617105960 ),
  { 4330  163 }  (         0.721, 5.96264301567,  12559.03815298200 ),
  { 4330  164 }  (         0.966, 2.74714939953,   6244.94281435360 ),
  { 4330  165 }  (         0.921, 0.10155275926,  29088.81141598500 ),
  { 4330  166 }  (         0.692, 3.89764447548,   1589.07289528380 ),
  { 4330  167 }  (         0.719, 5.91791450402,   4136.91043351620 ),
  { 4330  168 }  (         0.772, 4.05505682353,   6127.65545055720 ),
  { 4330  169 }  (         0.712, 5.49291532439,  22003.91463486980 ),
  { 4330  170 }  (         0.672, 1.60700490811,  11087.28512591840 ),
  { 4330  171 }  (         0.690, 4.50539825563,    426.59819087600 ),
  { 4330  172 }  (         0.854, 3.26104981596,  20426.57109242200 ),
  { 4330  173 }  (         0.656, 4.32410182940,  16858.48253293320 ),
  { 4330  174 }  (         0.840, 2.59572585222,  28766.92442448400 ),
  { 4330  175 }  (         0.692, 0.61650089011,  11403.67699557500 ),
  { 4330  176 }  (         0.700, 3.40901167143,      7.11354700080 ),
  { 4330  177 }  (         0.726, 0.04243053594,   5481.25491886760 ),
  { 4330  178 }  (         0.557, 4.78317696534,  20199.09495963300 ),
  { 4330  179 }  (         0.649, 1.04027912958,   6062.66320755260 ),
  { 4330  180 }  (         0.633, 5.70229959167,  45892.73043315699 ),
  { 4330  181 }  (         0.592, 6.11836729658,   9623.68827669120 ),
  { 4330  182 }  (         0.523, 3.62840021266,   5333.90024102160 ),
  { 4330  183 }  (         0.604, 5.57734696185,  10344.29506538580 ),
  { 4330  184 }  (         0.496, 2.21023499449,   1990.74501704100 ),
  { 4330  185 }  (         0.691, 1.96071732602,  12416.58850284820 ),
  { 4330  186 }  (         0.640, 1.59074172032,  18319.53658487960 ),
  { 4330  187 }  (         0.625, 3.82362791378,  13517.87010623340 ),
  { 4330  188 }  (         0.663, 5.08444996779,    283.85931886520 ),
  { 4330  189 }  (         0.475, 1.17025894287,  12569.67481833180 ),
  { 4330  190 }  (         0.664, 4.50029469969,  47162.51635463520 ),
  { 4330  191 }  (         0.569, 0.16310365162,  17267.26820169119 ),
  { 4330  192 }  (         0.568, 3.86100969474,   6076.89030155420 ),
  { 4330  193 }  (         0.539, 4.83282276086,  18422.62935909819 ),
  { 4330  194 }  (         0.466, 0.75872342878,   7342.45778018060 ),
  { 4330  195 }  (         0.541, 3.07212190507, 226858.23855437008 ),
  { 4330  196 }  (         0.458, 0.26774483096,   4590.91018048900 ),
  { 4330  197 }  (         0.610, 1.53597051291,  33019.02111220460 ),
  { 4330  198 }  (         0.617, 2.62356328726,  11190.37790013700 ),
  { 4330  199 }  (         0.548, 4.55798855791,  18875.52586977400 ),
  { 4330  200 }  (         0.633, 4.60110281228,  66567.48586525429 ),
  { 4330  201 }  (         0.596, 5.78202396722,    632.78373931320 ),
  { 4330  202 }  (         0.533, 5.01786882904,  12132.43996210600 ),
  { 4330  203 }  (         0.603, 5.38458554802, 316428.22867391503 ),
  { 4330  204 }  (         0.469, 0.59168241917,  21954.15760939799 ),
  { 4330  205 }  (         0.548, 3.50613163558,  17253.04110768959 ),
  { 4330  206 }  (         0.502, 0.98804327589,  11609.86254401220 ),
  { 4330  207 }  (         0.568, 1.98497313089,   7668.63742494250 ),
  { 4330  208 }  (         0.482, 1.62141803864,  12146.66705610760 ),
  { 4330  209 }  (         0.391, 3.68718382989,  18052.92954315780 ),
  { 4330  210 }  (         0.457, 3.77205737340, 156137.47598479928 ),
  { 4330  211 }  (         0.401, 5.28260651958,  15671.08175940660 ),
  { 4330  212 }  (         0.469, 1.80963184268,  12562.62858163380 ),
  { 4330  213 }  (         0.508, 3.36399024699,  20597.24396304120 ),
  { 4330  214 }  (         0.450, 5.66054299250,  10454.50138660520 ),
  { 4330  215 }  (         0.375, 4.98534633105,   9779.10867612540 ),
  { 4330  216 }  (         0.523, 0.97215560834, 155427.54293624099 ),
  { 4330  217 }  (         0.403, 5.13939866506,   1551.04522264800 ),
  { 4330  218 }  (         0.372, 3.69883738807,   9388.00590941520 ),
  { 4330  219 }  (         0.367, 4.43875659716,   4535.05943692440 ),
  { 4330  220 }  (         0.406, 4.20863156600,  12592.45001978260 ),
  { 4330  221 }  (         0.360, 2.53924644657,    242.72860397400 ),
  { 4330  222 }  (         0.471, 4.61907324819,   5436.99301524020 ),
  { 4330  223 }  (         0.441, 5.83872966262,   3496.03282613400 ),
  { 4330  224 }  (         0.385, 4.94496680973,  24356.78078864160 ),
  { 4330  225 }  (         0.349, 6.15018231784,  19800.94595622480 ),
  { 4330  226 }  (         0.355, 0.21895678106,   5429.87946823940 ),
  { 4330  227 }  (         0.344, 5.62993724928,   2379.16447357160 ),
  { 4330  228 }  (         0.380, 2.72105213143,  11933.36796066960 ),
  { 4330  229 }  (         0.432, 0.24221790536,  17996.03116822220 ),
  { 4330  230 }  (         0.378, 5.22517556974,   7477.52286021600 ),
  { 4330  231 }  (         0.337, 5.10888041439,   5849.36411211460 ),
  { 4330  232 }  (         0.315, 0.57827745123,  10557.59416082380 ),
  { 4330  233 }  (         0.318, 4.49953141399,   3634.62102451840 ),
  { 4330  234 }  (         0.323, 1.54274281393,  10440.27429260360 ),
  { 4330  235 }  (         0.309, 5.76839284397,     20.77539549240 ),
  { 4330  236 }  (         0.301, 2.34727604008,   4686.88940770680 ),
  { 4330  237 }  (         0.414, 5.93237602310,  51092.72605085480 ),
  { 4330  238 }  (         0.361, 2.16398609550,  28237.23345938940 ),
  { 4330  239 }  (         0.288, 0.18376252189,  13095.84266507740 ),
  { 4330  240 }  (         0.277, 5.12952205045,  13119.72110282519 ),
  { 4330  241 }  (         0.327, 6.19222146204,   6268.84875598980 ),
  { 4330  242 }  (         0.273, 0.30522428863,  23141.55838292460 ),
  { 4330  243 }  (         0.267, 5.76152585786,   5966.68398033480 ),
  { 4330  244 }  (         0.308, 5.99280509979,  22805.73556599360 ),
  { 4330  245 }  (         0.345, 2.92489919444,  36949.23080842420 ),
  { 4330  246 }  (         0.253, 5.20995219509,  24072.92146977640 ),
  { 4330  247 }  (         0.342, 5.72702586209,  16460.33352952499 ),
  { 4330  248 }  (         0.261, 2.00304796059,   6148.01076995600 ),
  { 4330  249 }  (         0.238, 5.08264392839,   6915.85958930460 ),
  { 4330  250 }  (         0.249, 2.94762789744,    135.06508003540 ),
  { 4330  251 }  (         0.306, 3.89764686987,  10988.80815753500 ),
  { 4330  252 }  (         0.305, 0.05827812117,   4701.11650170840 ),
  { 4330  253 }  (         0.319, 2.95712862064, 163096.18036118349 ),
  { 4330  254 }  (         0.209, 4.43768461442,   6546.15977336420 ),
  { 4330  255 }  (         0.270, 2.06643178717,   4804.20927592700 ),
  { 4330  256 }  (         0.217, 0.73691592312,   6303.85124548380 ),
  { 4330  257 }  (         0.206, 0.32075959415,  25934.12433108940 ),
  { 4330  258 }  (         0.218, 0.18428135264,  28286.99048486120 ),
  { 4330  259 }  (         0.205, 5.21312087405,  20995.39296644940 ),
  { 4330  260 }  (         0.199, 0.44384292491,  16737.57723659660 ),
  { 4330  261 }  (         0.230, 6.06567392849,   6287.00800325450 ),
  { 4330  262 }  (         0.219, 1.29194216300,   5326.78669402080 ),
  { 4330  263 }  (         0.201, 1.74700937253,  22743.40937951640 ),
  { 4330  264 }  (         0.207, 4.45440927276,   6279.48542133960 ),
  { 4330  265 }  (         0.269, 6.05640445030,  64471.99124174489 ),
  { 4330  266 }  (         0.190, 0.99256176518,  29296.61538957860 ),
  { 4330  267 }  (         0.238, 5.42471431221,  39609.65458316560 ),
  { 4330  268 }  (         0.262, 5.26961924198,    522.57741809380 ),
  { 4330  269 }  (         0.210, 4.68618183158,   6254.62666252360 ),
  { 4330  270 }  (         0.197, 2.80624554080,   4933.20844033260 ),
  { 4330  271 }  (         0.252, 4.36220154608,  40879.44050464380 ),
  { 4330  272 }  (         0.261, 1.07241516738,  55022.93574707440 ),
  { 4330  273 }  (         0.189, 3.82966734476,    419.48464387520 ),
  { 4330  274 }  (         0.185, 4.14324541379,   5642.19824260920 ),
  { 4330  275 }  (         0.247, 3.44855612987,   6702.56049386660 ),
  { 4330  276 }  (         0.205, 4.04424043223,    536.80451209540 ),
  { 4330  277 }  (         0.191, 3.14082686083,  16723.35014259500 ),
  { 4330  278 }  (         0.222, 5.16263907319,  23539.70738633280 ),
  { 4330  279 }  (         0.180, 4.56214752149,   6489.26139842860 ),
  { 4330  280 }  (         0.219, 0.80382553358,  16627.37091537720 ),
  { 4330  281 }  (         0.227, 0.60156339452,   5905.70224207560 ),
  { 4330  282 }  (         0.168, 0.88753528161,  16062.18452611680 ),
  { 4330  283 }  (         0.158, 0.92127725775,  23937.85638974100 ),
  { 4330  284 }  (         0.157, 4.69607868164,   6805.65326808520 ),
  { 4330  285 }  (         0.207, 4.88410451334,   6286.66627864320 ),
  { 4330  286 }  (         0.160, 4.95943826846,  10021.83728009940 ),
  { 4330  287 }  (         0.166, 0.97126433565,   3097.88382272579 ),
  { 4330  288 }  (         0.209, 5.75663411805,   3646.35037735440 ),
  { 4330  289 }  (         0.175, 6.12762824412, 239424.39025435288 ),
  { 4330  290 }  (         0.173, 3.13887234973,   6179.98307577280 ),
  { 4330  291 }  (         0.157, 3.62822058179,  18451.07854656599 ),
  { 4330  292 }  (         0.157, 4.67695912235,   6709.67404086740 ),
  { 4330  293 }  (         0.146, 3.09506069735,   4907.30205014560 ),
  { 4330  294 }  (         0.165, 2.27139128760,  10660.68693504240 ),
  { 4330  295 }  (         0.201, 1.67701267433,   2107.03450754240 ),
  { 4330  296 }  (         0.144, 3.96947747592,   6019.99192661860 ),
  { 4330  297 }  (         0.171, 5.91302216729,   6058.73105428950 ),
  { 4330  298 }  (         0.144, 2.13155655120,  26084.02180621620 ),
  { 4330  299 }  (         0.151, 0.67417383554,   2388.89402044920 ),
  { 4330  300 }  (         0.189, 5.07122281033,    263.08392337280 ),
  { 4330  301 }  (         0.146, 5.10373877968,  10770.89325626180 ),
  { 4330  302 }  (         0.187, 1.23915444627,  19402.79695281660 ),
  { 4330  303 }  (         0.174, 0.08407293391,   9380.95967271720 ),
  { 4330  304 }  (         0.137, 1.26247412309,  12566.21901028560 ),
  { 4330  305 }  (         0.137, 3.52826010842,    639.89728631400 ),
  { 4330  306 }  (         0.148, 1.76124372592,   5888.44996493220 ),
  { 4330  307 }  (         0.164, 2.39195095081,   6357.85744855870 ),
  { 4330  308 }  (         0.146, 2.43675816553,   5881.40372823420 ),
  { 4330  309 }  (         0.161, 1.15721259372,  26735.94526221320 ),
  { 4330  310 }  (         0.131, 2.51859277344,   6599.46771964800 ),
  { 4330  311 }  (         0.153, 5.85203687779,   6281.59137728310 ),
  { 4330  312 }  (         0.151, 3.72338532649,  12669.24447420140 ),
  { 4330  313 }  (         0.132, 2.38417741883,   6525.80445396540 ),
  { 4330  314 }  (         0.129, 0.75556744143,   5017.50837136500 ),
  { 4330  315 }  (         0.127, 0.00254936441,  10027.90319572920 ),
  { 4330  316 }  (         0.148, 2.85102145528,   6418.14093002680 ),
  { 4330  317 }  (         0.143, 5.74460279367,  26087.90314157420 ),
  { 4330  318 }  (         0.172, 0.41289962240, 174242.46596404970 ),
  { 4330  319 }  (         0.136, 4.15497742275,   6311.52503745920 ),
  { 4330  320 }  (         0.170, 5.98194913129, 327574.51427678125 ),
  { 4330  321 }  (         0.124, 1.65497607604,  32217.20018108080 ),
  { 4330  322 }  (         0.136, 2.48430783417,  13341.67431130680 ),
  { 4330  323 }  (         0.165, 2.49667924600,  58953.14544329400 ),
  { 4330  324 }  (         0.123, 3.45660563754,   6277.55292568400 ),
  { 4330  325 }  (         0.117, 0.86065134175,   6245.04817735560 ),
  { 4330  326 }  (         0.149, 5.61358280963,   5729.50644714900 ),
  { 4330  327 }  (         0.153, 0.26860029950,    245.83164622940 ),
  { 4330  328 }  (         0.128, 0.71204006588,    103.09277421860 ),
  { 4330  329 }  (         0.159, 2.43166592149, 221995.02880149524 ),
  { 4330  330 }  (         0.130, 2.80707316718,   6016.46880826960 ),
  { 4330  331 }  (         0.137, 1.70657709294,  12566.08438968000 ),
  { 4330  332 }  (         0.111, 1.56305648432,  17782.73207278420 ),
  { 4330  333 }  (         0.113, 3.58302904101,  25685.87280280800 ),
  { 4330  334 }  (         0.109, 3.26403795962,   6819.88036208680 ),
  { 4330  335 }  (         0.122, 0.34120688217,   1162.47470440780 ),
  { 4330  336 }  (         0.119, 5.84644718278,  12721.57209941700 ),
  { 4330  337 }  (         0.144, 2.28899679126,  12489.88562870720 ),
  { 4330  338 }  (         0.137, 5.82029768354,  44809.65020086340 ),
  { 4330  339 }  (         0.107, 2.42818544140,   5547.19933645960 ),
  { 4330  340 }  (         0.134, 1.26539982939,   5331.35744374080 ),
  { 4330  341 }  (         0.103, 5.96518130595,   6321.10352262720 ),
  { 4330  342 }  (         0.109, 0.33808549034,  11300.58422135640 ),
  { 4330  343 }  (         0.129, 5.89187277327,  12029.34718788740 ),
  { 4330  344 }  (         0.122, 5.77325634636,  11919.14086666800 ),
  { 4330  345 }  (         0.107, 6.24998989350,  77690.75950573849 ),
  { 4330  346 }  (         0.107, 1.00535580713,  77736.78343050249 ),
  { 4330  347 }  (         0.143, 0.24122178432,   4214.06901508480 ),
  { 4330  348 }  (         0.143, 0.88529649733,   7576.56007357400 ),
  { 4330  349 }  (         0.107, 2.92124030496,  31415.37924995700 ),
  { 4330  350 }  (         0.099, 5.70862227072,   5540.08578945880 ),
  { 4330  351 }  (         0.110, 0.37528037383,   5863.59120611620 ),
  { 4330  352 }  (         0.104, 4.44107178366,   2118.76386037840 ),
  { 4330  353 }  (         0.098, 5.95877916706,   4061.21921539440 ),
  { 4330  354 }  (         0.113, 1.24206857385,  84672.47584450469 ),
  { 4330  355 }  (         0.124, 2.55619029867,  12539.85338018300 ),
  { 4330  356 }  (         0.110, 3.66952094329, 238004.52415723629 ),
  { 4330  357 }  (         0.112, 4.32512422943,  97238.62754448749 ),
  { 4330  358 }  (         0.097, 3.70151541181,  11720.06886523160 ),
  { 4330  359 }  (         0.120, 1.26895630252,  12043.57428188900 ),
  { 4330  360 }  (         0.094, 2.56461130309,  19004.64794940840 ),
  { 4330  361 }  (         0.117, 3.65425622684,  34520.30930938080 ),
  { 4330  362 }  (         0.098, 0.13589994287,  11080.17157891760 ),
  { 4330  363 }  (         0.097, 5.38330115253,   7834.12107263940 ),
  { 4330  364 }  (         0.097, 2.46722096722,  71980.63357473118 ),
  { 4330  365 }  (         0.095, 5.36958330451,   6288.59877429880 ),
  { 4330  366 }  (         0.111, 5.01961920313,  11823.16163945020 ),
  { 4330  367 }  (         0.090, 2.72299804525,  26880.31981303260 ),
  { 4330  368 }  (         0.099, 0.90164266377,  18635.92845453620 ),
  { 4330  369 }  (         0.126, 4.78722177847, 305281.94307104882 ),
  { 4330  370 }  (         0.093, 0.21240380046,  18139.29450141590 ),
  { 4330  371 }  (         0.124, 5.00979495566, 172146.97134054029 ),
  { 4330  372 }  (         0.099, 5.67090026475,  16522.65971600220 ),
  { 4330  373 }  (         0.092, 2.28180963676,  12491.37010141550 ),
  { 4330  374 }  (         0.090, 4.50544881196,  40077.61957352000 ),
  { 4330  375 }  (         0.100, 2.00639461612,  12323.42309600880 ),
  { 4330  376 }  (         0.095, 5.68801979087,  14919.01785375460 ),
  { 4330  377 }  (         0.087, 1.86043406047,  27707.54249429480 ),
  { 4330  378 }  (         0.105, 3.02903468417,  22345.26037610820 ),
  { 4330  379 }  (         0.087, 5.43970168638,   6272.03014972750 ),
  { 4330  380 }  (         0.089, 1.63389387182,  33326.57873317420 ),
  { 4330  381 }  (         0.082, 5.58298993353,  10241.20229116720 ),
  { 4330  382 }  (         0.094, 5.47749711149,   9924.81042151060 ),
  { 4330  383 }  (         0.082, 4.71988314145,  15141.39079431200 ),
  { 4330  384 }  (         0.097, 5.61458778738,   2787.04302385740 ),
  { 4330  385 }  (         0.096, 3.89073946348,   6379.05507720920 ),
  { 4330  386 }  (         0.081, 3.13038482444,  36147.40987730040 ),
  { 4330  387 }  (         0.110, 4.89978492291,  72140.62866668739 ),
  { 4330  388 }  (         0.097, 5.20764563059,   6303.43116939020 ),
  { 4330  389 }  (         0.082, 5.26342716139,   9814.60410029120 ),
  { 4330  390 }  (         0.109, 2.35555589770,  83286.91426955358 ),
  { 4330  391 }  (         0.097, 2.58492958057,  30666.15495843280 ),
  { 4330  392 }  (         0.093, 1.32651591333,  23020.65308658799 ),
  { 4330  393 }  (         0.078, 3.99588630754,  11293.47067435560 ),
  { 4330  394 }  (         0.090, 0.57771932738,  26482.17080962440 ),
  { 4330  395 }  (         0.106, 3.92012705073,  62883.35513951360 ),
  { 4330  396 }  (         0.098, 2.94397773524,    316.39186965660 ),
  { 4330  397 }  (         0.076, 3.96310417608,  29026.48522950779 ),
  { 4330  398 }  (         0.078, 1.97068529306,  90279.92316810328 ),
  { 4330  399 }  (         0.076, 0.23027966596,  21424.46664430340 ),
  { 4330  400 }  (         0.080, 2.23099742212,    266.60704172180 ),
  { 4330  401 }  (         0.079, 1.46227790922,   8982.81066930900 ),
  { 4330  402 }  (         0.102, 4.92129953565,   5621.84292321040 ),
  { 4330  403 }  (         0.100, 0.39243148321,  24279.10701821359 ),
  { 4330  404 }  (         0.071, 1.52014858474,  33794.54372352860 ),
  { 4330  405 }  (         0.076, 0.22880641443,  57375.80190084620 ),
  { 4330  406 }  (         0.091, 0.96515913904,  48739.85989708300 ),
  { 4330  407 }  (         0.075, 2.77638585157,  12964.30070339100 ),
  { 4330  408 }  (         0.077, 5.18846946344,  11520.99686379520 ),
  { 4330  409 }  (         0.068, 0.50006599129,   4274.51831083240 ),
  { 4330  410 }  (         0.075, 2.07323762803,  15664.03552270859 ),
  { 4330  411 }  (         0.074, 1.01884134928,   6393.28217121080 ),
  { 4330  412 }  (         0.077, 0.46665178780,  16207.88627150200 ),
  { 4330  413 }  (         0.081, 4.10452219483, 161710.61878623239 ),
  { 4330  414 }  (         0.067, 3.83840630887,   6262.72053059260 ),
  { 4330  415 }  (         0.071, 3.91415523291,   7875.67186362420 ),
  { 4330  416 }  (         0.081, 0.91938383237,     74.78159856730 ),
  { 4330  417 }  (         0.083, 4.69916218791,  23006.42599258639 ),
  { 4330  418 }  (         0.063, 2.32556465878,   6279.19451463340 ),
  { 4330  419 }  (         0.065, 5.41938745446,  28628.33622609960 ),
  { 4330  420 }  (         0.065, 3.02336771694,   5959.57043333400 ),
  { 4330  421 }  (         0.064, 3.31033198370,   2636.72547263700 ),
  { 4330  422 }  (         0.064, 0.18375587519,   1066.49547719000 ),
  { 4330  423 }  (         0.080, 5.81239171612,  12341.80690428090 ),
  { 4330  424 }  (         0.066, 2.15105504851,     38.02767263580 ),
  { 4330  425 }  (         0.062, 2.43313614978,  10138.10951694860 ),
  { 4330  426 }  (         0.060, 3.16153906470,   5490.30096152400 ),
  { 4330  427 }  (         0.069, 0.30764736334,   7018.95236352320 ),
  { 4330  428 }  (         0.068, 2.24442548639,  24383.07910844140 ),
  { 4330  429 }  (         0.078, 1.39649386463,   9411.46461508720 ),
  { 4330  430 }  (         0.063, 0.72976362625,   6286.95718534940 ),
  { 4330  431 }  (         0.073, 4.95125917731,   6453.74872061060 ),
  { 4330  432 }  (         0.078, 0.32736023459,   6528.90749622080 ),
  { 4330  433 }  (         0.059, 4.95362151577,  35707.71008290740 ),
  { 4330  434 }  (         0.070, 2.37962727525,  15508.61512327440 ),
  { 4330  435 }  (         0.073, 1.35229143111,   5327.47610838280 ),
  { 4330  436 }  (         0.072, 5.91833527334,  10881.09957748120 ),
  { 4330  437 }  (         0.059, 5.36231868425,  10239.58386601080 ),
  { 4330  438 }  (         0.059, 1.63156134967,  61306.01159706580 ),
  { 4330  439 }  (         0.054, 4.29491690425,  21947.11137270000 ),
  { 4330  440 }  (         0.057, 5.89190132575,  34513.26307268280 ),
  { 4330  441 }  (         0.074, 1.38235845304,   9967.45389998160 ),
  { 4330  442 }  (         0.053, 3.86543309344,  32370.97899156560 ),
  { 4330  443 }  (         0.055, 4.51794544854,  34911.41207609100 ),
  { 4330  444 }  (         0.063, 5.41479412056,  11502.83761653050 ),
  { 4330  445 }  (         0.063, 2.34416220742,  11510.70192305670 ),
  { 4330  446 }  (         0.068, 0.77493931112,  29864.33402730900 ),
  { 4330  447 }  (         0.060, 5.57024703495,   5756.90800324580 ),
  { 4330  448 }  (         0.072, 2.80863088166,  10866.87248347960 ),
  { 4330  449 }  (         0.061, 2.69736991384,  82576.98122099529 ),
  { 4330  450 }  (         0.063, 5.32068807257,   3116.65941225980 ),
  { 4330  451 }  (         0.052, 1.02278758099,   6272.43918464160 ),
  { 4330  452 }  (         0.069, 5.00698550308,  25287.72379939980 ),
  { 4330  453 }  (         0.066, 6.12047940728,  12074.48840752400 ),
  { 4330  454 }  (         0.051, 2.59519527563,  11396.56344857420 ),
  { 4330  455 }  (         0.056, 2.57995973521,  17892.93839400359 ),
  { 4330  456 }  (         0.059, 0.44167237620, 250570.67585721909 ),
  { 4330  457 }  (         0.059, 3.84070143543,   5483.25472482600 ),
  { 4330  458 }  (         0.049, 0.54704693048,  22594.05489571199 ),
  { 4330  459 }  (         0.065, 2.38423614501,  52670.06959330260 ),
  { 4330  460 }  (         0.069, 5.34363738671,  66813.56483573320 ),
  { 4330  461 }  (         0.057, 5.42770501007, 310145.15282392364 ),
  { 4330  462 }  (         0.053, 1.17760296075,    149.56319713460 ),
  { 4330  463 }  (         0.061, 4.02090887211,  34596.36465465240 ),
  { 4330  464 }  (         0.049, 4.18361320516,  18606.49894600020 ),
  { 4330  465 }  (         0.055, 0.83886167974,  20452.86941222180 ),
  { 4330  466 }  (         0.050, 1.46327331958,  37455.72649597440 ),
  { 4330  467 }  (         0.048, 4.53854727167,  29822.78323632420 ),
  { 4330  468 }  (         0.058, 3.34847975377,  33990.61834428620 ),
  { 4330  469 }  (         0.065, 1.45522693982,  76251.32777062019 ),
  { 4330  470 }  (         0.056, 2.35650663692,  37724.75341974820 ),
  { 4330  471 }  (         0.052, 2.61551081496,   5999.21653112620 ),
  { 4330  472 }  (         0.053, 0.17334326094,  77717.29458646949 ),
  { 4330  473 }  (         0.053, 0.79879700631,  77710.24834977149 ),
  { 4330  474 }  (         0.047, 0.43240779709,    735.87651353180 ),
  { 4330  475 }  (         0.053, 4.58763261686,  11616.97609101300 ),
  { 4330  476 }  (         0.048, 6.20230111054,   4171.42553661380 ),
  { 4330  477 }  (         0.052, 1.09723616404,    640.87760738220 ),
  { 4330  478 }  (         0.057, 3.42008310383,  50317.20343953080 ),
  { 4330  479 }  (         0.053, 1.01528448581, 149144.46708624958 ),
  { 4330  480 }  (         0.047, 3.00924906195,  52175.80628314840 ),
  { 4330  481 }  (         0.052, 2.03254070404,   6293.71251534120 ),
  { 4330  482 }  (         0.048, 0.12356889734,  13362.44970679920 ),
  { 4330  483 }  (         0.045, 3.37963782356,  10763.77970926100 ),
  { 4330  484 }  (         0.047, 5.50981287869,  12779.45079542080 ),
  { 4330  485 }  (         0.062, 5.45209070099,    949.17560896980 ),
  { 4330  486 }  (         0.061, 2.93237974631,   5791.41255753260 ),
  { 4330  487 }  (         0.044, 2.87440620802,   8584.66166590080 ),
  { 4330  488 }  (         0.046, 4.03141796560,  10667.80048204320 ),
  { 4330  489 }  (         0.047, 3.89902931422,   3903.91137641980 ),
  { 4330  490 }  (         0.046, 2.75700467329,   6993.00889854970 ),
  { 4330  491 }  (         0.045, 1.93386293300,    206.18554843720 ),
  { 4330  492 }  (         0.047, 2.57670800912,  11492.54267579200 ),
  { 4330  493 }  (         0.044, 3.62570223167,  63658.87775083760 ),
  { 4330  494 }  (         0.051, 0.84536826273,  12345.73905754400 ),
  { 4330  495 }  (         0.043, 0.01524970172,  37853.87549938260 ),
  { 4330  496 }  (         0.041, 3.27146326065,   8858.31494432060 ),
  { 4330  497 }  (         0.045, 3.03765521215,  65236.22129328540 ),
  { 4330  498 }  (         0.047, 1.44447548944,  21393.54196985760 ),
  { 4330  499 }  (         0.058, 5.45843180927,   1975.49254585600 ),
  { 4330  500 }  (         0.050, 2.13285524146,  12573.26524698360 ),
  { 4330  501 }  (         0.041, 1.32190847146,   2547.83753823240 ),
  { 4330  502 }  (         0.047, 3.67579608544,  28313.28880466100 ),
  { 4330  503 }  (         0.041, 2.24013475126,   8273.82086703240 ),
  { 4330  504 }  (         0.047, 6.21438985953,  10991.30589870060 ),
  { 4330  505 }  (         0.042, 3.01631817350,    853.19638175200 ),
  { 4330  506 }  (         0.056, 1.09773690181,  77376.20102240759 ),
  { 4330  507 }  (         0.040, 2.35698541041,   2699.73481931760 ),
  { 4330  508 }  (         0.043, 5.28030898459,  17796.95916678580 ),
  { 4330  509 }  (         0.054, 2.59175932091,  22910.44676536859 ),
  { 4330  510 }  (         0.054, 0.88027764102,  71960.38658322369 ),
  { 4330  511 }  (         0.055, 0.07988899477,  83467.15635301729 ),
  { 4330  512 }  (         0.039, 1.12867321442,   9910.58332750900 ),
  { 4330  513 }  (         0.040, 1.35670430524,  27177.85152920020 ),
  { 4330  514 }  (         0.039, 4.39624220245,   5618.31980486140 ),
  { 4330  515 }  (         0.042, 4.78798367468,   7856.89627409019 ),
  { 4330  516 }  (         0.047, 2.75482175292,  18202.21671665939 ),
  { 4330  517 }  (         0.039, 1.97008298629,  24491.42579258340 ),
  { 4330  518 }  (         0.042, 4.04346599946,   7863.94251078820 ),
  { 4330  519 }  (         0.038, 0.49178679251,  38650.17350619900 ),
  { 4330  520 }  (         0.036, 4.86047906533,   4157.19844261220 ),
  { 4330  521 }  (         0.043, 5.64354880978,   1062.90504853820 ),
  { 4330  522 }  (         0.036, 3.98066313627,  12565.17137891460 ),
  { 4330  523 }  (         0.042, 2.30753932657,   6549.68289171320 ),
  { 4330  524 }  (         0.040, 5.39694918320,   9498.21223063460 ),
  { 4330  525 }  (         0.040, 3.30603243754,  23536.11695768099 ),
  { 4330  526 }  (         0.050, 6.15760345261,  78051.34191383338 )
  (*$endif *)
      );

  (*$ifdef meeus *)
  vsop87_ear_r1:array[0..  9,0..2] of extended = (
  (*$else *)
  vsop87_ear_r1:array[0..291,0..2] of extended = (
  (*$endif *)
  { 4331    1 }  (    103018.608, 1.10748969588,   6283.07584999140 ),
  { 4331    2 }  (      1721.238, 1.06442301418,  12566.15169998280 ),
  { 4331    3 }  (       702.215, 3.14159265359,      0.00000000000 ),
  { 4331    4 }  (        32.346, 1.02169059149,  18849.22754997420 ),
  { 4331    5 }  (        30.799, 2.84353804832,   5507.55323866740 ),
  { 4331    6 }  (        24.971, 1.31906709482,   5223.69391980220 ),
  { 4331    7 }  (        18.485, 1.42429748614,   1577.34354244780 ),
  { 4331    8 }  (        10.078, 5.91378194648,  10977.07880469900 ),
  { 4331    9 }  (         8.634, 0.27146150602,   5486.77784317500 ),
  { 4331   10 }  (         8.654, 1.42046854427,   6275.96230299060 )
  (*$ifndef meeus *)                                                 ,
  { 4331   11 }  (         5.069, 1.68613426734,   5088.62883976680 ),
  { 4331   12 }  (         4.985, 6.01401770704,   6286.59896834040 ),
  { 4331   13 }  (         4.669, 5.98724494073,    529.69096509460 ),
  { 4331   14 }  (         4.395, 0.51800238019,   4694.00295470760 ),
  { 4331   15 }  (         3.872, 4.74969833437,   2544.31441988340 ),
  { 4331   16 }  (         3.750, 5.07097685568,    796.29800681640 ),
  { 4331   17 }  (         4.100, 1.08424786092,   9437.76293488700 ),
  { 4331   18 }  (         3.518, 0.02290216272,  83996.84731811189 ),
  { 4331   19 }  (         3.436, 0.94937019624,  71430.69561812909 ),
  { 4331   20 }  (         3.221, 6.15628775313,   2146.16541647520 ),
  { 4331   21 }  (         3.414, 5.41218322538,    775.52261132400 ),
  { 4331   22 }  (         2.863, 5.48432847146,  10447.38783960440 ),
  { 4331   23 }  (         2.520, 0.24276941146,    398.14900340820 ),
  { 4331   24 }  (         2.201, 4.95216196651,   6812.76681508600 ),
  { 4331   25 }  (         2.186, 0.41991743105,   8031.09226305840 ),
  { 4331   26 }  (         2.838, 3.42034351366,   2352.86615377180 ),
  { 4331   27 }  (         2.554, 6.13241878525,   6438.49624942560 ),
  { 4331   28 }  (         1.932, 5.31374608366,   8429.24126646660 ),
  { 4331   29 }  (         2.429, 3.09164528262,   4690.47983635860 ),
  { 4331   30 }  (         1.730, 1.53686208550,   4705.73230754360 ),
  { 4331   31 }  (         2.250, 3.68863633842,   7084.89678111520 ),
  { 4331   32 }  (         2.093, 1.28191783032,   1748.01641306700 ),
  { 4331   33 }  (         1.441, 0.81656250862,  14143.49524243060 ),
  { 4331   34 }  (         1.483, 3.22225357771,   7234.79425624200 ),
  { 4331   35 }  (         1.754, 3.22883705112,   6279.55273164240 ),
  { 4331   36 }  (         1.583, 4.09702349428,  11499.65622279280 ),
  { 4331   37 }  (         1.575, 5.53890170575,   3154.68708489560 ),
  { 4331   38 }  (         1.847, 1.82040335363,   7632.94325965020 ),
  { 4331   39 }  (         1.504, 3.63293385726,  11513.88331679440 ),
  { 4331   40 }  (         1.337, 4.64440864339,   6836.64525283380 ),
  { 4331   41 }  (         1.275, 2.69341415363,   1349.86740965880 ),
  { 4331   42 }  (         1.352, 6.15101580257,   5746.27133789600 ),
  { 4331   43 }  (         1.125, 3.35673439497,  17789.84561978500 ),
  { 4331   44 }  (         1.470, 3.65282991755,   1194.44701022460 ),
  { 4331   45 }  (         1.177, 2.57676109092,  13367.97263110660 ),
  { 4331   46 }  (         1.101, 4.49748696552,   4292.33083295040 ),
  { 4331   47 }  (         1.234, 5.65036509521,   5760.49843189760 ),
  { 4331   48 }  (         0.984, 0.65517395136,   5856.47765911540 ),
  { 4331   49 }  (         0.928, 2.32420318751,  10213.28554621100 ),
  { 4331   50 }  (         1.077, 5.82812169132,  12036.46073488820 ),
  { 4331   51 }  (         0.916, 0.76613009583,  16730.46368959580 ),
  { 4331   52 }  (         0.877, 1.50137505051,  11926.25441366880 ),
  { 4331   53 }  (         1.023, 5.62076589825,   6256.77753019160 ),
  { 4331   54 }  (         0.851, 0.65709335533,    155.42039943420 ),
  { 4331   55 }  (         0.802, 4.10519132088,    951.71840625060 ),
  { 4331   56 }  (         0.857, 1.41661697538,   5753.38488489680 ),
  { 4331   57 }  (         0.994, 1.14418521187,   1059.38193018920 ),
  { 4331   58 }  (         0.813, 1.63948433322,   6681.22485339960 ),
  { 4331   59 }  (         0.662, 4.55200452260,   5216.58037280140 ),
  { 4331   60 }  (         0.644, 4.19478168733,   6040.34724601740 ),
  { 4331   61 }  (         0.626, 1.50767713598,   5643.17856367740 ),
  { 4331   62 }  (         0.590, 6.18277145205,   4164.31198961300 ),
  { 4331   63 }  (         0.635, 0.52413263542,   6290.18939699220 ),
  { 4331   64 }  (         0.650, 0.97935690350,  25132.30339996560 ),
  { 4331   65 }  (         0.568, 2.30125315873,  10973.55568635000 ),
  { 4331   66 }  (         0.547, 5.27256412213,   3340.61242669980 ),
  { 4331   67 }  (         0.547, 2.20144422886,   1592.59601363280 ),
  { 4331   68 }  (         0.526, 0.92464258226,  11371.70468975820 ),
  { 4331   69 }  (         0.490, 5.90951388655,   3894.18182954220 ),
  { 4331   70 }  (         0.478, 1.66857963179,  12168.00269657460 ),
  { 4331   71 }  (         0.516, 3.59803483887,  10969.96525769820 ),
  { 4331   72 }  (         0.518, 3.97914412373,  17298.18232732620 ),
  { 4331   73 }  (         0.534, 5.03740926442,   9917.69687450980 ),
  { 4331   74 }  (         0.487, 2.50545369269,   6127.65545055720 ),
  { 4331   75 }  (         0.416, 4.04828175503,  10984.19235169980 ),
  { 4331   76 }  (         0.538, 5.54081539805,    553.56940284240 ),
  { 4331   77 }  (         0.402, 2.16544019233,   7860.41939243920 ),
  { 4331   78 }  (         0.553, 2.32177369366,  11506.76976979360 ),
  { 4331   79 }  (         0.367, 3.39152532250,   6496.37494542940 ),
  { 4331   80 }  (         0.360, 5.34379853282,   7079.37385680780 ),
  { 4331   81 }  (         0.337, 3.61563704045,  11790.62908865880 ),
  { 4331   82 }  (         0.456, 0.30754294809,    801.82093112380 ),
  { 4331   83 }  (         0.417, 3.70009308674,  10575.40668294180 ),
  { 4331   84 }  (         0.381, 5.82033971802,   7058.59846131540 ),
  { 4331   85 }  (         0.321, 0.31988767355,  16200.77272450120 ),
  { 4331   86 }  (         0.364, 1.08414306177,   6309.37416979120 ),
  { 4331   87 }  (         0.294, 4.54798604957,  11856.21865142450 ),
  { 4331   88 }  (         0.290, 1.26473978562,   8635.94200376320 ),
  { 4331   89 }  (         0.399, 4.16998866302,     26.29831979980 ),
  { 4331   90 }  (         0.262, 5.08316906342,  10177.25767953360 ),
  { 4331   91 }  (         0.243, 2.25746091190,  11712.95531823080 ),
  { 4331   92 }  (         0.237, 1.05070575346,    242.72860397400 ),
  { 4331   93 }  (         0.275, 3.45319481756,   5884.92684658320 ),
  { 4331   94 }  (         0.255, 5.38496831087,  21228.39202354580 ),
  { 4331   95 }  (         0.307, 4.24313526604,   3738.76143010800 ),
  { 4331   96 }  (         0.216, 3.46037894728,    213.29909543800 ),
  { 4331   97 }  (         0.196, 0.69029243914,   1990.74501704100 ),
  { 4331   98 }  (         0.198, 5.16301829964,  12352.85260454480 ),
  { 4331   99 }  (         0.214, 3.91876200279,  13916.01910964160 ),
  { 4331  100 }  (         0.212, 4.00861198517,   5230.80746680300 ),
  { 4331  101 }  (         0.184, 5.59805976614,   6283.14316029419 ),
  { 4331  102 }  (         0.184, 2.85275392124,   7238.67559160000 ),
  { 4331  103 }  (         0.179, 2.54259058334,  14314.16811304980 ),
  { 4331  104 }  (         0.225, 1.64458698399,   4732.03062734340 ),
  { 4331  105 }  (         0.236, 5.58826125715,   6069.77675455340 ),
  { 4331  106 }  (         0.187, 2.72805985443,   6062.66320755260 ),
  { 4331  107 }  (         0.184, 6.04216273598,   6283.00853968860 ),
  { 4331  108 }  (         0.230, 3.62591335086,   6284.05617105960 ),
  { 4331  109 }  (         0.163, 2.19117396803,  18073.70493865020 ),
  { 4331  110 }  (         0.172, 0.97612950740,   3930.20969621960 ),
  { 4331  111 }  (         0.215, 1.04672844028,   3496.03282613400 ),
  { 4331  112 }  (         0.169, 4.75084479006,  17267.26820169119 ),
  { 4331  113 }  (         0.152, 0.19390712179,   9779.10867612540 ),
  { 4331  114 }  (         0.182, 5.16288118255,  17253.04110768959 ),
  { 4331  115 }  (         0.149, 0.80944184260,    709.93304855830 ),
  { 4331  116 }  (         0.163, 2.19209570390,   6076.89030155420 ),
  { 4331  117 }  (         0.186, 5.01159497089,  11015.10647733480 ),
  { 4331  118 }  (         0.134, 0.97765485759,  65147.61976813770 ),
  { 4331  119 }  (         0.141, 4.38421981312,   4136.91043351620 ),
  { 4331  120 }  (         0.158, 4.60974280627,   9623.68827669120 ),
  { 4331  121 }  (         0.133, 3.30508592837, 154717.60988768269 ),
  { 4331  122 }  (         0.163, 6.11782626245,      3.52311834900 ),
  { 4331  123 }  (         0.174, 1.58078542187,      7.11354700080 ),
  { 4331  124 }  (         0.141, 0.49976927274,  25158.60171976540 ),
  { 4331  125 }  (         0.124, 6.03440460031,   9225.53927328300 ),
  { 4331  126 }  (         0.150, 5.30166336812,  13517.87010623340 ),
  { 4331  127 }  (         0.127, 1.92389511438,  22483.84857449259 ),
  { 4331  128 }  (         0.121, 2.37813129011, 167283.76158766549 ),
  { 4331  129 }  (         0.120, 3.98423684853,   4686.88940770680 ),
  { 4331  130 }  (         0.117, 5.81072642211,  12569.67481833180 ),
  { 4331  131 }  (         0.122, 5.60973054224,   5642.19824260920 ),
  { 4331  132 }  (         0.157, 3.40236426002,  16496.36139620240 ),
  { 4331  133 }  (         0.129, 2.10705116371,   1589.07289528380 ),
  { 4331  134 }  (         0.116, 0.55839966736,   5849.36411211460 ),
  { 4331  135 }  (         0.123, 1.52961392771,  12559.03815298200 ),
  { 4331  136 }  (         0.111, 0.44848279675,   6172.86952877200 ),
  { 4331  137 }  (         0.123, 5.81645568991,   6282.09552892320 ),
  { 4331  138 }  (         0.150, 4.26278409223,   3128.38876509580 ),
  { 4331  139 }  (         0.106, 2.27437761356,   5429.87946823940 ),
  { 4331  140 }  (         0.104, 4.42743707728,  23543.23050468179 ),
  { 4331  141 }  (         0.121, 0.39459045915,  12132.43996210600 ),
  { 4331  142 }  (         0.104, 2.41842602527,    426.59819087600 ),
  { 4331  143 }  (         0.110, 5.80381480447,  16858.48253293320 ),
  { 4331  144 }  (         0.100, 2.93805577485,   4535.05943692440 ),
  { 4331  145 }  (         0.097, 3.97935904984,   6133.51265285680 ),
  { 4331  146 }  (         0.110, 6.22339014386,  12146.66705610760 ),
  { 4331  147 }  (         0.098, 0.87576563709,   6525.80445396540 ),
  { 4331  148 }  (         0.098, 3.15248421301,  10440.27429260360 ),
  { 4331  149 }  (         0.095, 2.46168411100,   3097.88382272579 ),
  { 4331  150 }  (         0.088, 0.23371480284,  13119.72110282519 ),
  { 4331  151 }  (         0.098, 5.77016493489,   7342.45778018060 ),
  { 4331  152 }  (         0.092, 6.03915555063,  20426.57109242200 ),
  { 4331  153 }  (         0.096, 5.56909292561,   2388.89402044920 ),
  { 4331  154 }  (         0.081, 1.32131147691,   5650.29211067820 ),
  { 4331  155 }  (         0.086, 3.94529200528,  10454.50138660520 ),
  { 4331  156 }  (         0.076, 2.70729716925, 143571.32428481648 ),
  { 4331  157 }  (         0.091, 5.64100034152,   8827.39026987480 ),
  { 4331  158 }  (         0.076, 1.80783856698,  28286.99048486120 ),
  { 4331  159 }  (         0.081, 1.90858992196,  29088.81141598500 ),
  { 4331  160 }  (         0.075, 3.40955892978,   5481.25491886760 ),
  { 4331  161 }  (         0.069, 4.49936170873,  17256.63153634140 ),
  { 4331  162 }  (         0.088, 1.10098454357,  11769.85369316640 ),
  { 4331  163 }  (         0.066, 2.78285801977,    536.80451209540 ),
  { 4331  164 }  (         0.068, 3.88179770758,  17260.15465469040 ),
  { 4331  165 }  (         0.084, 1.59303306354,   9380.95967271720 ),
  { 4331  166 }  (         0.088, 3.88076636762,   7477.52286021600 ),
  { 4331  167 }  (         0.061, 6.17558202197,  11087.28512591840 ),
  { 4331  168 }  (         0.060, 4.34824715818,   6206.80977871580 ),
  { 4331  169 }  (         0.082, 4.59843208943,   9388.00590941520 ),
  { 4331  170 }  (         0.079, 1.63131230601,   4933.20844033260 ),
  { 4331  171 }  (         0.078, 4.20905757484,   5729.50644714900 ),
  { 4331  172 }  (         0.057, 5.48157926651,  18319.53658487960 ),
  { 4331  173 }  (         0.060, 1.01261781084,  12721.57209941700 ),
  { 4331  174 }  (         0.056, 1.63031935692,  15720.83878487840 ),
  { 4331  175 }  (         0.055, 0.24926735018,  15110.46611986620 ),
  { 4331  176 }  (         0.061, 5.93059279661,  12539.85338018300 ),
  { 4331  177 }  (         0.055, 4.84298966314,  13095.84266507740 ),
  { 4331  178 }  (         0.067, 6.11690589247,   8662.24032356300 ),
  { 4331  179 }  (         0.054, 5.73750638571,   3634.62102451840 ),
  { 4331  180 }  (         0.074, 1.05466745829,  16460.33352952499 ),
  { 4331  181 }  (         0.053, 2.29084335688,  16062.18452611680 ),
  { 4331  182 }  (         0.064, 2.13513767927,   7875.67186362420 ),
  { 4331  183 }  (         0.067, 0.07096807518,  14945.31617355440 ),
  { 4331  184 }  (         0.051, 2.31511194429,   6262.72053059260 ),
  { 4331  185 }  (         0.057, 5.77055471237,  12043.57428188900 ),
  { 4331  186 }  (         0.056, 4.41980790431,   4701.11650170840 ),
  { 4331  187 }  (         0.059, 5.87963500073,   5331.35744374080 ),
  { 4331  188 }  (         0.058, 2.30546168628,    955.59974160860 ),
  { 4331  189 }  (         0.049, 1.93839278478,   5333.90024102160 ),
  { 4331  190 }  (         0.048, 2.69973662261,   6709.67404086740 ),
  { 4331  191 }  (         0.064, 1.64379897981,   6262.30045449900 ),
  { 4331  192 }  (         0.046, 3.98449608961,  98068.53671630539 ),
  { 4331  193 }  (         0.050, 3.68875893005,  12323.42309600880 ),
  { 4331  194 }  (         0.045, 3.30068569697,  22003.91463486980 ),
  { 4331  195 }  (         0.047, 1.26317154881,  11919.14086666800 ),
  { 4331  196 }  (         0.045, 0.89150445122,  51868.24866217880 ),
  { 4331  197 }  (         0.043, 1.61526242998,   6277.55292568400 ),
  { 4331  198 }  (         0.043, 5.74295325645,  11403.67699557500 ),
  { 4331  199 }  (         0.044, 3.43070646822,  10021.83728009940 ),
  { 4331  200 }  (         0.056, 0.02481833774,  15671.08175940660 ),
  { 4331  201 }  (         0.055, 3.14274403422,  33019.02111220460 ),
  { 4331  202 }  (         0.045, 3.00877289177,   8982.81066930900 ),
  { 4331  203 }  (         0.046, 0.73303568429,   6303.43116939020 ),
  { 4331  204 }  (         0.049, 1.60455690285,   6303.85124548380 ),
  { 4331  205 }  (         0.045, 0.40210030323,   6805.65326808520 ),
  { 4331  206 }  (         0.053, 0.94869680175,  10988.80815753500 ),
  { 4331  207 }  (         0.041, 1.61122384329,   6819.88036208680 ),
  { 4331  208 }  (         0.055, 0.89439119424,  11933.36796066960 ),
  { 4331  209 }  (         0.045, 3.88495384656,  60530.48898574180 ),
  { 4331  210 }  (         0.040, 4.75740908001,  38526.57435087200 ),
  { 4331  211 }  (         0.040, 1.49921251887,  18451.07854656599 ),
  { 4331  212 }  (         0.040, 3.77498297228,  26087.90314157420 ),
  { 4331  213 }  (         0.051, 1.70258603562,   1551.04522264800 ),
  { 4331  214 }  (         0.039, 2.97100699926,   2118.76386037840 ),
  { 4331  215 }  (         0.053, 5.19854123078,  77713.77146812050 ),
  { 4331  216 }  (         0.047, 4.26356628717,  21424.46664430340 ),
  { 4331  217 }  (         0.037, 0.62902722802,  24356.78078864160 ),
  { 4331  218 }  (         0.036, 0.11087914947,  10344.29506538580 ),
  { 4331  219 }  (         0.036, 0.77037556319,  12029.34718788740 ),
  { 4331  220 }  (         0.035, 3.30933994515,  24072.92146977640 ),
  { 4331  221 }  (         0.035, 5.93650887012,  31570.79964939120 ),
  { 4331  222 }  (         0.036, 2.15108874765,  30774.50164257480 ),
  { 4331  223 }  (         0.036, 1.75078825382,  16207.88627150200 ),
  { 4331  224 }  (         0.033, 5.06264177921, 226858.23855437008 ),
  { 4331  225 }  (         0.034, 6.16891378800,  24491.42579258340 ),
  { 4331  226 }  (         0.035, 3.19120695549,  32217.20018108080 ),
  { 4331  227 }  (         0.034, 2.31528650443,  55798.45835839840 ),
  { 4331  228 }  (         0.032, 4.21446357042,  15664.03552270859 ),
  { 4331  229 }  (         0.039, 1.24979117796,   6418.14093002680 ),
  { 4331  230 }  (         0.037, 4.11943655770,   2787.04302385740 ),
  { 4331  231 }  (         0.032, 1.62887710890,    639.89728631400 ),
  { 4331  232 }  (         0.038, 5.89832942685,    640.87760738220 ),
  { 4331  233 }  (         0.032, 1.72442327688,  27433.88921587499 ),
  { 4331  234 }  (         0.031, 2.78828943753,  12139.55350910680 ),
  { 4331  235 }  (         0.035, 4.44608896525,  18202.21671665939 ),
  { 4331  236 }  (         0.034, 3.96287980676,  18216.44381066100 ),
  { 4331  237 }  (         0.033, 4.73611335874,  16723.35014259500 ),
  { 4331  238 }  (         0.034, 1.43910280005,  49515.38250840700 ),
  { 4331  239 }  (         0.031, 0.23302920161,  23581.25817731760 ),
  { 4331  240 }  (         0.029, 2.02633840220,  11609.86254401220 ),
  { 4331  241 }  (         0.030, 2.54923230240,   9924.81042151060 ),
  { 4331  242 }  (         0.032, 4.91793198558,  11300.58422135640 ),
  { 4331  243 }  (         0.028, 0.26187189577,  13521.75144159140 ),
  { 4331  244 }  (         0.028, 3.84568936822,   2699.73481931760 ),
  { 4331  245 }  (         0.029, 1.83149729794,  29822.78323632420 ),
  { 4331  246 }  (         0.033, 4.60320094415,  19004.64794940840 ),
  { 4331  247 }  (         0.027, 4.46183450287,   6702.56049386660 ),
  { 4331  248 }  (         0.030, 4.46494072240,  36147.40987730040 ),
  { 4331  249 }  (         0.027, 0.03211931363,   6279.78949257360 ),
  { 4331  250 }  (         0.026, 5.46497324333,   6245.04817735560 ),
  { 4331  251 }  (         0.035, 4.52695674113,  36949.23080842420 ),
  { 4331  252 }  (         0.027, 3.52528177609,  10770.89325626180 ),
  { 4331  253 }  (         0.026, 1.48499438453,  11080.17157891760 ),
  { 4331  254 }  (         0.035, 2.82154380962,  19402.79695281660 ),
  { 4331  255 }  (         0.025, 2.46339998836,   6279.48542133960 ),
  { 4331  256 }  (         0.026, 4.97688894643,  16737.57723659660 ),
  { 4331  257 }  (         0.026, 2.36136541526,  17996.03116822220 ),
  { 4331  258 }  (         0.029, 4.15148654061,  45892.73043315699 ),
  { 4331  259 }  (         0.026, 4.50714272714,  17796.95916678580 ),
  { 4331  260 }  (         0.027, 4.72625223674,   1066.49547719000 ),
  { 4331  261 }  (         0.025, 2.89309528854,   6286.66627864320 ),
  { 4331  262 }  (         0.027, 0.37462444357,  12964.30070339100 ),
  { 4331  263 }  (         0.029, 4.94860010533,   5863.59120611620 ),
  { 4331  264 }  (         0.031, 3.93096113577,  29864.33402730900 ),
  { 4331  265 }  (         0.024, 6.14987193584,  18606.49894600020 ),
  { 4331  266 }  (         0.024, 3.74225964547,  29026.48522950779 ),
  { 4331  267 }  (         0.025, 5.70460621565,  27707.54249429480 ),
  { 4331  268 }  (         0.025, 5.33928840652,  15141.39079431200 ),
  { 4331  269 }  (         0.027, 3.02320897140,   6286.36220740920 ),
  { 4331  270 }  (         0.023, 0.28364955406,   5327.47610838280 ),
  { 4331  271 }  (         0.026, 1.34240461687,  18875.52586977400 ),
  { 4331  272 }  (         0.024, 1.33998410121,  19800.94595622480 ),
  { 4331  273 }  (         0.025, 6.00172494004,   6489.26139842860 ),
  { 4331  274 }  (         0.022, 1.81777974484,   6288.59877429880 ),
  { 4331  275 }  (         0.022, 3.58603606640,   6915.85958930460 ),
  { 4331  276 }  (         0.029, 2.09564449439,  15265.88651930040 ),
  { 4331  277 }  (         0.022, 1.02173599251,  11925.27409260060 ),
  { 4331  278 }  (         0.022, 4.74660932338,  28230.18722269139 ),
  { 4331  279 }  (         0.021, 2.30688751432,   5999.21653112620 ),
  { 4331  280 }  (         0.021, 3.22654944430,  25934.12433108940 ),
  { 4331  281 }  (         0.021, 3.04956726238,   6566.93516885660 ),
  { 4331  282 }  (         0.027, 5.35653084499,  33794.54372352860 ),
  { 4331  283 }  (         0.028, 3.91168324815,  18208.34994259200 ),
  { 4331  284 }  (         0.020, 1.52296293311,    135.06508003540 ),
  { 4331  285 }  (         0.022, 4.66462839521,  13362.44970679920 ),
  { 4331  286 }  (         0.019, 1.78121167862, 156137.47598479928 ),
  { 4331  287 }  (         0.019, 2.99969102221,  19651.04848109800 ),
  { 4331  288 }  (         0.019, 2.86664273362,  18422.62935909819 ),
  { 4331  289 }  (         0.025, 0.94995632141,  31415.37924995700 ),
  { 4331  290 }  (         0.019, 4.71432851499,  77690.75950573849 ),
  { 4331  291 }  (         0.019, 2.54227398241,  77736.78343050249 ),
  { 4331  292 }  (         0.020, 5.91915117116,  48739.85989708300 )
  (*$endif *)
      );
  (*$ifdef meeus *)
  vsop87_ear_r2:array[0..  5,0..2] of extended = (
  (*$else *)
  vsop87_ear_r2:array[0..138,0..2] of extended = (
  (*$endif *)
  { 4332    1 }  (      4359.385, 5.78455133738,   6283.07584999140 ),
  { 4332    2 }  (       123.633, 5.57934722157,  12566.15169998280 ),
  { 4332    3 }  (        12.341, 3.14159265359,      0.00000000000 ),
  { 4332    4 }  (         8.792, 3.62777733395,  77713.77146812050 ),
  { 4332    5 }  (         5.689, 1.86958905084,   5573.14280143310 ),
  { 4332    6 }  (         3.301, 5.47027913302,  18849.22754997420 )
  (*$ifndef meeus *)                                                 ,
  { 4332    7 }  (         1.471, 4.48028885617,   5507.55323866740 ),
  { 4332    8 }  (         1.013, 2.81456417694,   5223.69391980220 ),
  { 4332    9 }  (         0.854, 3.10878241236,   1577.34354244780 ),
  { 4332   10 }  (         1.102, 2.84173992403, 161000.68573767410 ),
  { 4332   11 }  (         0.648, 5.47349498544,    775.52261132400 ),
  { 4332   12 }  (         0.609, 1.37969434104,   6438.49624942560 ),
  { 4332   13 }  (         0.499, 4.41649242250,   6286.59896834040 ),
  { 4332   14 }  (         0.417, 0.90242451175,  10977.07880469900 ),
  { 4332   15 }  (         0.402, 3.20376585290,   5088.62883976680 ),
  { 4332   16 }  (         0.351, 1.81079227770,   5486.77784317500 ),
  { 4332   17 }  (         0.467, 3.65753702738,   7084.89678111520 ),
  { 4332   18 }  (         0.458, 5.38585314743, 149854.40013480789 ),
  { 4332   19 }  (         0.304, 3.51701098693,    796.29800681640 ),
  { 4332   20 }  (         0.266, 6.17413982699,   6836.64525283380 ),
  { 4332   21 }  (         0.279, 1.84120501086,   4694.00295470760 ),
  { 4332   22 }  (         0.260, 1.41629543251,   2146.16541647520 ),
  { 4332   23 }  (         0.266, 3.13832905677,  71430.69561812909 ),
  { 4332   24 }  (         0.321, 5.35313367048,   3154.68708489560 ),
  { 4332   25 }  (         0.238, 2.17720020018,    155.42039943420 ),
  { 4332   26 }  (         0.293, 4.61501268144,   4690.47983635860 ),
  { 4332   27 }  (         0.229, 4.75969588070,   7234.79425624200 ),
  { 4332   28 }  (         0.211, 0.21868065485,   4705.73230754360 ),
  { 4332   29 }  (         0.201, 4.21905743357,   1349.86740965880 ),
  { 4332   30 }  (         0.195, 4.57808285364,    529.69096509460 ),
  { 4332   31 }  (         0.253, 2.81496293039,   1748.01641306700 ),
  { 4332   32 }  (         0.182, 5.70454011389,   6040.34724601740 ),
  { 4332   33 }  (         0.179, 6.02897097053,   4292.33083295040 ),
  { 4332   34 }  (         0.186, 1.58690991244,   6309.37416979120 ),
  { 4332   35 }  (         0.170, 2.90220009715,   9437.76293488700 ),
  { 4332   36 }  (         0.166, 1.99984925026,   8031.09226305840 ),
  { 4332   37 }  (         0.158, 0.04783713552,   2544.31441988340 ),
  { 4332   38 }  (         0.197, 2.01083639502,   1194.44701022460 ),
  { 4332   39 }  (         0.165, 5.78372596778,  83996.84731811189 ),
  { 4332   40 }  (         0.214, 3.38285934319,   7632.94325965020 ),
  { 4332   41 }  (         0.140, 0.36401486094,  10447.38783960440 ),
  { 4332   42 }  (         0.151, 0.95153163031,   6127.65545055720 ),
  { 4332   43 }  (         0.136, 1.48426306582,   2352.86615377180 ),
  { 4332   44 }  (         0.127, 5.48475435134,    951.71840625060 ),
  { 4332   45 }  (         0.126, 5.26866506592,   6279.55273164240 ),
  { 4332   46 }  (         0.125, 3.75754889288,   6812.76681508600 ),
  { 4332   47 }  (         0.101, 4.95015746147,    398.14900340820 ),
  { 4332   48 }  (         0.102, 0.68468295277,   1592.59601363280 ),
  { 4332   49 }  (         0.100, 1.14568935785,   3894.18182954220 ),
  { 4332   50 }  (         0.129, 0.76540016965,    553.56940284240 ),
  { 4332   51 }  (         0.109, 5.41063597567,   6256.77753019160 ),
  { 4332   52 }  (         0.075, 5.84804322893,    242.72860397400 ),
  { 4332   53 }  (         0.095, 1.94452244083,  11856.21865142450 ),
  { 4332   54 }  (         0.077, 0.69373708195,   8429.24126646660 ),
  { 4332   55 }  (         0.100, 5.19725292131, 244287.60000722769 ),
  { 4332   56 }  (         0.080, 6.18440483705,   1059.38193018920 ),
  { 4332   57 }  (         0.069, 5.25699888595,  14143.49524243060 ),
  { 4332   58 }  (         0.085, 5.39484725499,  25132.30339996560 ),
  { 4332   59 }  (         0.066, 0.51779993906,    801.82093112380 ),
  { 4332   60 }  (         0.055, 5.16878202461,   7058.59846131540 ),
  { 4332   61 }  (         0.051, 3.88759155247,  12036.46073488820 ),
  { 4332   62 }  (         0.050, 5.57636570536,   6290.18939699220 ),
  { 4332   63 }  (         0.061, 2.24359003264,   8635.94200376320 ),
  { 4332   64 }  (         0.050, 5.54441900966,   1990.74501704100 ),
  { 4332   65 }  (         0.056, 4.00301078040,  13367.97263110660 ),
  { 4332   66 }  (         0.052, 4.13138898038,   7860.41939243920 ),
  { 4332   67 }  (         0.052, 3.90943054011,     26.29831979980 ),
  { 4332   68 }  (         0.041, 3.57128482780,   7079.37385680780 ),
  { 4332   69 }  (         0.056, 2.76959005761,  90955.55169449610 ),
  { 4332   70 }  (         0.042, 1.91461189199,   7477.52286021600 ),
  { 4332   71 }  (         0.042, 0.42728171713,  10213.28554621100 ),
  { 4332   72 }  (         0.042, 1.09413724455,    709.93304855830 ),
  { 4332   73 }  (         0.039, 3.93298068961,  10973.55568635000 ),
  { 4332   74 }  (         0.038, 6.17935925345,   9917.69687450980 ),
  { 4332   75 }  (         0.049, 0.83021145241,  11506.76976979360 ),
  { 4332   76 }  (         0.053, 1.45828359397, 233141.31440436149 ),
  { 4332   77 }  (         0.047, 6.21568666789,   6681.22485339960 ),
  { 4332   78 }  (         0.037, 0.36359309980,  10177.25767953360 ),
  { 4332   79 }  (         0.035, 3.33024911524,   5643.17856367740 ),
  { 4332   80 }  (         0.034, 5.63446915337,   6525.80445396540 ),
  { 4332   81 }  (         0.035, 5.36033855038,  25158.60171976540 ),
  { 4332   82 }  (         0.034, 5.36319798321,   4933.20844033260 ),
  { 4332   83 }  (         0.033, 4.24722336872,  12569.67481833180 ),
  { 4332   84 }  (         0.043, 5.26370903404,  10575.40668294180 ),
  { 4332   85 }  (         0.042, 5.08837645072,  11015.10647733480 ),
  { 4332   86 }  (         0.040, 1.98334703186,   6284.05617105960 ),
  { 4332   87 }  (         0.042, 4.22496037505,  88860.05707098669 ),
  { 4332   88 }  (         0.029, 3.19088628170,  11926.25441366880 ),
  { 4332   89 }  (         0.029, 0.15217616684,  12168.00269657460 ),
  { 4332   90 }  (         0.030, 1.61904744136,   9779.10867612540 ),
  { 4332   91 }  (         0.027, 0.76388991416,   1589.07289528380 ),
  { 4332   92 }  (         0.036, 2.74712003443,   3738.76143010800 ),
  { 4332   93 }  (         0.033, 3.08807829566,   3930.20969621960 ),
  { 4332   94 }  (         0.031, 5.34906619513, 143571.32428481648 ),
  { 4332   95 }  (         0.025, 0.10240267494,  22483.84857449259 ),
  { 4332   96 }  (         0.030, 3.47110495524,  14945.31617355440 ),
  { 4332   97 }  (         0.024, 1.10425016019,   4535.05943692440 ),
  { 4332   98 }  (         0.024, 1.58037259780,   6496.37494542940 ),
  { 4332   99 }  (         0.023, 3.87710321433,   6275.96230299060 ),
  { 4332  100 }  (         0.025, 3.94529778970,   3128.38876509580 ),
  { 4332  101 }  (         0.023, 3.44685609601,   4136.91043351620 ),
  { 4332  102 }  (         0.023, 3.83156029849,   5753.38488489680 ),
  { 4332  103 }  (         0.022, 1.86956128067,  16730.46368959580 ),
  { 4332  104 }  (         0.025, 2.42188933855,   5729.50644714900 ),
  { 4332  105 }  (         0.020, 1.78208352927,  17789.84561978500 ),
  { 4332  106 }  (         0.021, 4.30363087400,  16858.48253293320 ),
  { 4332  107 }  (         0.021, 0.49258939822,  29088.81141598500 ),
  { 4332  108 }  (         0.025, 1.33030250444,   6282.09552892320 ),
  { 4332  109 }  (         0.027, 2.54785812264,   3496.03282613400 ),
  { 4332  110 }  (         0.022, 1.11232521950,  12721.57209941700 ),
  { 4332  111 }  (         0.021, 5.97759081637,      7.11354700080 ),
  { 4332  112 }  (         0.019, 0.80292033311,  16062.18452611680 ),
  { 4332  113 }  (         0.023, 4.12454848769,   2388.89402044920 ),
  { 4332  114 }  (         0.022, 4.92663152168,  18875.52586977400 ),
  { 4332  115 }  (         0.023, 5.68902059771,  16460.33352952499 ),
  { 4332  116 }  (         0.023, 4.97346265647,  17260.15465469040 ),
  { 4332  117 }  (         0.023, 3.03021283729,  66567.48586525429 ),
  { 4332  118 }  (         0.016, 3.89740925257,   5331.35744374080 ),
  { 4332  119 }  (         0.017, 3.08268671348, 154717.60988768269 ),
  { 4332  120 }  (         0.016, 3.95085099736,   3097.88382272579 ),
  { 4332  121 }  (         0.016, 3.99041783945,   6283.14316029419 ),
  { 4332  122 }  (         0.020, 6.10644140189, 167283.76158766549 ),
  { 4332  123 }  (         0.015, 4.09775914607,  11712.95531823080 ),
  { 4332  124 }  (         0.016, 5.71769940700,  17298.18232732620 ),
  { 4332  125 }  (         0.016, 3.28894009404,   5884.92684658320 ),
  { 4332  126 }  (         0.015, 5.64785377164,  12559.03815298200 ),
  { 4332  127 }  (         0.016, 4.43452080930,   6283.00853968860 ),
  { 4332  128 }  (         0.014, 2.31721603062,   5481.25491886760 ),
  { 4332  129 }  (         0.014, 4.43479032305,  13517.87010623340 ),
  { 4332  130 }  (         0.014, 4.73209312936,   7342.45778018060 ),
  { 4332  131 }  (         0.012, 0.64705975463,  18073.70493865020 ),
  { 4332  132 }  (         0.011, 1.51443332200,  16200.77272450120 ),
  { 4332  133 }  (         0.011, 0.88708889185,  21228.39202354580 ),
  { 4332  134 }  (         0.014, 4.50116508534,    640.87760738220 ),
  { 4332  135 }  (         0.011, 4.64339996198,  11790.62908865880 ),
  { 4332  136 }  (         0.011, 1.31064298246,   4164.31198961300 ),
  { 4332  137 }  (         0.009, 3.02238989305,  23543.23050468179 ),
  { 4332  138 }  (         0.009, 2.04999402381,  22003.91463486980 ),
  { 4332  139 }  (         0.009, 4.91488110218,    213.29909543800 )
  (*$endif *)
      );
  (*$ifdef meeus *)
  vsop87_ear_r3:array[0..  1,0..2] of extended = (
  (*$else *)
  vsop87_ear_r3:array[0.. 26,0..2] of extended = (
  (*$endif *)
  { 4333    1 }  (       144.595, 4.27319435148,   6283.07584999140 ),
  { 4333    2 }  (         6.729, 3.91697608662,  12566.15169998280 )
  (*$ifndef meeus *)                                                 ,
  { 4333    3 }  (         0.774, 0.00000000000,      0.00000000000 ),
  { 4333    4 }  (         0.247, 3.73019298781,  18849.22754997420 ),
  { 4333    5 }  (         0.036, 2.80081409050,   6286.59896834040 ),
  { 4333    6 }  (         0.033, 5.62216602775,   6127.65545055720 ),
  { 4333    7 }  (         0.019, 3.71292621802,   6438.49624942560 ),
  { 4333    8 }  (         0.016, 4.26011484232,   6525.80445396540 ),
  { 4333    9 }  (         0.016, 3.50416887054,   6256.77753019160 ),
  { 4333   10 }  (         0.014, 3.62127621114,  25132.30339996560 ),
  { 4333   11 }  (         0.011, 4.39200958819,   4705.73230754360 ),
  { 4333   12 }  (         0.011, 5.22327127059,   6040.34724601740 ),
  { 4333   13 }  (         0.010, 4.28045254647,  83996.84731811189 ),
  { 4333   14 }  (         0.009, 1.56864096494,   5507.55323866740 ),
  { 4333   15 }  (         0.011, 1.37795688024,   6309.37416979120 ),
  { 4333   16 }  (         0.010, 5.19937959068,  71430.69561812909 ),
  { 4333   17 }  (         0.009, 0.47275199930,   6279.55273164240 ),
  { 4333   18 }  (         0.009, 0.74642756529,   5729.50644714900 ),
  { 4333   19 }  (         0.007, 2.97374891560,    775.52261132400 ),
  { 4333   20 }  (         0.007, 3.28615691021,   7058.59846131540 ),
  { 4333   21 }  (         0.007, 2.19184402142,   6812.76681508600 ),
  { 4333   22 }  (         0.005, 3.15419034438,    529.69096509460 ),
  { 4333   23 }  (         0.006, 4.54725567047,   1059.38193018920 ),
  { 4333   24 }  (         0.005, 1.51104406936,   7079.37385680780 ),
  { 4333   25 }  (         0.007, 2.98052059053,   6681.22485339960 ),
  { 4333   26 }  (         0.005, 2.30961231391,  12036.46073488820 ),
  { 4333   27 }  (         0.005, 3.71102966917,   6290.18939699220 )
  (*$endif *)
      );
  (*$ifdef meeus *)
  vsop87_ear_r4:array[0..  0,0..2] of extended = (
  (*$else *)
  vsop87_ear_r4:array[0..  9,0..2] of extended = (
  (*$endif *)
  { 4334    1 }  (         3.858, 2.56384387339,   6283.07584999140 )
  (*$ifndef meeus *)                                                 ,
  { 4334    2 }  (         0.306, 2.26769501230,  12566.15169998280 ),
  { 4334    3 }  (         0.053, 3.44031471924,   5573.14280143310 ),
  { 4334    4 }  (         0.015, 2.04794573436,  18849.22754997420 ),
  { 4334    5 }  (         0.013, 2.05688873673,  77713.77146812050 ),
  { 4334    6 }  (         0.007, 4.41218854480, 161000.68573767410 ),
  { 4334    7 }  (         0.005, 5.26154653107,   6438.49624942560 ),
  { 4334    8 }  (         0.005, 4.07695126049,   6127.65545055720 ),
  { 4334    9 }  (         0.006, 3.81514213664, 149854.40013480789 ),
  { 4334   10 }  (         0.003, 1.28175749811,   6286.59896834040 )
  (*$endif *)
      );
  (*$ifdef meeus *)
  vsop87_ear_r5:array[0..  0,0..2] of extended = (
  (*$else *)
  vsop87_ear_r5:array[0..  2,0..2] of extended = (
  (*$endif *)
  (*$ifdef meeus *)
                 (         0.000, 0.00000000000,      0.00000000000 )
  (*$else *)
  { 4335    1 }  (         0.086, 1.21579741687,   6283.07584999140 ),
  { 4335    2 }  (         0.012, 0.65617264033,  12566.15169998280 ),
  { 4335    3 }  (         0.001, 0.38068797142,  18849.22754997420 )
  (*$endif *)
      );
begin
  WITH result do begin
    a:=0; b:=0; c:=0;
    case index of
      0: if (nr>=low(vsop87_ear_r0)) and (nr<=high(vsop87_ear_r0)) then begin
        a:=vsop87_ear_r0[nr,0];  b:=vsop87_ear_r0[nr,1];  c:=vsop87_ear_r0[nr,2];
        end;
      1: if (nr>=low(vsop87_ear_r1)) and (nr<=high(vsop87_ear_r1)) then begin
        a:=vsop87_ear_r1[nr,0];  b:=vsop87_ear_r1[nr,1];  c:=vsop87_ear_r1[nr,2];
        end;
      2: if (nr>=low(vsop87_ear_r2)) and (nr<=high(vsop87_ear_r2)) then begin
        a:=vsop87_ear_r2[nr,0];  b:=vsop87_ear_r2[nr,1];  c:=vsop87_ear_r2[nr,2];
        end;
      3: if (nr>=low(vsop87_ear_r3)) and (nr<=high(vsop87_ear_r3)) then begin
        a:=vsop87_ear_r3[nr,0];  b:=vsop87_ear_r3[nr,1];  c:=vsop87_ear_r3[nr,2];
        end;
      4: if (nr>=low(vsop87_ear_r4)) and (nr<=high(vsop87_ear_r4)) then begin
        a:=vsop87_ear_r4[nr,0];  b:=vsop87_ear_r4[nr,1];  c:=vsop87_ear_r4[nr,2];
        end;
      5: if (nr>=low(vsop87_ear_r5)) and (nr<=high(vsop87_ear_r5)) then begin
        a:=vsop87_ear_r5[nr,0];  b:=vsop87_ear_r5[nr,1];  c:=vsop87_ear_r5[nr,2];
        end;
      end;
    end;
  end;

function TVSOPEarth.LatitudeFactor(nr,index: integer):TVSOPEntry;
const
  (*$ifdef meeus *)
  vsop87_ear_b0:array[0..  4,0..2] of extended = (
  (*$else *)
  vsop87_ear_b0:array[0..183,0..2] of extended = (
  (*$endif *)
  { 4320    1 }  (       279.620, 3.19870156017,  84334.66158130829 ),
  { 4320    2 }  (       101.643, 5.42248619256,   5507.55323866740 ),
  { 4320    3 }  (        80.445, 3.88013204458,   5223.69391980220 ),
  { 4320    4 }  (        43.806, 3.70444689758,   2352.86615377180 ),
  { 4320    5 }  (        31.933, 4.00026369781,   1577.34354244780 )
  (*$ifndef meeus *)                                                 ,
  { 4320    6 }  (        22.724, 3.98473831560,   1047.74731175470 ),
  { 4320    7 }  (        16.392, 3.56456119782,   5856.47765911540 ),
  { 4320    8 }  (        18.141, 4.98367470263,   6283.07584999140 ),
  { 4320    9 }  (        14.443, 3.70275614914,   9437.76293488700 ),
  { 4320   10 }  (        14.304, 3.41117857525,  10213.28554621100 ),
  { 4320   11 }  (        11.246, 4.82820690530,  14143.49524243060 ),
  { 4320   12 }  (        10.900, 2.08574562327,   6812.76681508600 ),
  { 4320   13 }  (         9.714, 3.47303947752,   4694.00295470760 ),
  { 4320   14 }  (        10.367, 4.05663927946,  71092.88135493269 ),
  { 4320   15 }  (         8.775, 4.44016515669,   5753.38488489680 ),
  { 4320   16 }  (         8.366, 4.99251512180,   7084.89678111520 ),
  { 4320   17 }  (         6.921, 4.32559054073,   6275.96230299060 ),
  { 4320   18 }  (         9.145, 1.14182646613,   6620.89011318780 ),
  { 4320   19 }  (         7.194, 3.60193205752,    529.69096509460 ),
  { 4320   20 }  (         7.698, 5.55425745881, 167621.57585086189 ),
  { 4320   21 }  (         5.285, 2.48446991566,   4705.73230754360 ),
  { 4320   22 }  (         5.208, 6.24992674537,  18073.70493865020 ),
  { 4320   23 }  (         4.529, 2.33827747356,   6309.37416979120 ),
  { 4320   24 }  (         5.579, 4.41023653738,   7860.41939243920 ),
  { 4320   25 }  (         4.743, 0.70995680136,   5884.92684658320 ),
  { 4320   26 }  (         4.301, 1.10255777773,   6681.22485339960 ),
  { 4320   27 }  (         3.849, 1.82229412531,   5486.77784317500 ),
  { 4320   28 }  (         4.093, 5.11700141207,  13367.97263110660 ),
  { 4320   29 }  (         3.681, 0.43793170356,   3154.68708489560 ),
  { 4320   30 }  (         3.420, 5.42034800952,   6069.77675455340 ),
  { 4320   31 }  (         3.617, 6.04641937526,   3930.20969621960 ),
  { 4320   32 }  (         3.670, 4.58210192227,  12194.03291462090 ),
  { 4320   33 }  (         2.918, 1.95463881126,  10977.07880469900 ),
  { 4320   34 }  (         2.797, 5.61259275048,  11790.62908865880 ),
  { 4320   35 }  (         2.502, 0.60499729367,   6496.37494542940 ),
  { 4320   36 }  (         2.319, 5.01648216014,   1059.38193018920 ),
  { 4320   37 }  (         2.684, 1.39470396488,  22003.91463486980 ),
  { 4320   38 }  (         2.428, 3.24183056052,  78051.58573131690 ),
  { 4320   39 }  (         2.120, 4.30691000285,   5643.17856367740 ),
  { 4320   40 }  (         2.257, 3.15557225618,  90617.73743129970 ),
  { 4320   41 }  (         1.813, 3.75574218285,   3340.61242669980 ),
  { 4320   42 }  (         2.226, 2.79699346659,  12036.46073488820 ),
  { 4320   43 }  (         1.888, 0.86991545823,   8635.94200376320 ),
  { 4320   44 }  (         1.517, 1.95852055701,    398.14900340820 ),
  { 4320   45 }  (         1.581, 3.19976230948,   5088.62883976680 ),
  { 4320   46 }  (         1.421, 6.25530883827,   2544.31441988340 ),
  { 4320   47 }  (         1.595, 0.25619915135,  17298.18232732620 ),
  { 4320   48 }  (         1.391, 4.69964175561,   7058.59846131540 ),
  { 4320   49 }  (         1.478, 2.81808207569,  25934.12433108940 ),
  { 4320   50 }  (         1.481, 3.65823554806,  11506.76976979360 ),
  { 4320   51 }  (         1.693, 4.95689385293, 156475.29024799570 ),
  { 4320   52 }  (         1.183, 1.29343061246,    775.52261132400 ),
  { 4320   53 }  (         1.114, 2.37889311846,   3738.76143010800 ),
  { 4320   54 }  (         0.994, 4.30088900425,   9225.53927328300 ),
  { 4320   55 }  (         0.924, 3.06451026812,   4164.31198961300 ),
  { 4320   56 }  (         0.867, 0.55606931068,   8429.24126646660 ),
  { 4320   57 }  (         0.988, 5.97286104208,   7079.37385680780 ),
  { 4320   58 }  (         0.824, 1.50984806173,  10447.38783960440 ),
  { 4320   59 }  (         0.915, 0.12635654592,  11015.10647733480 ),
  { 4320   60 }  (         0.742, 1.99159139281,  26087.90314157420 ),
  { 4320   61 }  (         1.039, 3.14159265359,      0.00000000000 ),
  { 4320   62 }  (         0.850, 4.24120016095,  29864.33402730900 ),
  { 4320   63 }  (         0.755, 2.89631873320,   4732.03062734340 ),
  { 4320   64 }  (         0.714, 1.37548118603,   2146.16541647520 ),
  { 4320   65 }  (         0.708, 1.91406542362,   8031.09226305840 ),
  { 4320   66 }  (         0.746, 0.57893808616,    796.29800681640 ),
  { 4320   67 }  (         0.802, 5.12339137230,   2942.46342329160 ),
  { 4320   68 }  (         0.751, 1.67479850166,  21228.39202354580 ),
  { 4320   69 }  (         0.602, 4.09976538826,  64809.80550494129 ),
  { 4320   70 }  (         0.594, 3.49580704962,  16496.36139620240 ),
  { 4320   71 }  (         0.592, 4.59481504319,   4690.47983635860 ),
  { 4320   72 }  (         0.530, 5.73979295200,   8827.39026987480 ),
  { 4320   73 }  (         0.503, 5.66433137112,  33794.54372352860 ),
  { 4320   74 }  (         0.483, 1.57106522411,    801.82093112380 ),
  { 4320   75 }  (         0.438, 0.06707733767,   3128.38876509580 ),
  { 4320   76 }  (         0.423, 2.86944595927,  12566.15169998280 ),
  { 4320   77 }  (         0.504, 3.26207669160,   7632.94325965020 ),
  { 4320   78 }  (         0.552, 1.02926440457, 239762.20451754928 ),
  { 4320   79 }  (         0.427, 3.67434378210,    213.29909543800 ),
  { 4320   80 }  (         0.404, 1.46193297142,  15720.83878487840 ),
  { 4320   81 }  (         0.503, 4.85802444134,   6290.18939699220 ),
  { 4320   82 }  (         0.417, 0.81920713533,   5216.58037280140 ),
  { 4320   83 }  (         0.365, 0.01002966162,  12168.00269657460 ),
  { 4320   84 }  (         0.363, 1.28376436579,   6206.80977871580 ),
  { 4320   85 }  (         0.353, 4.70059133110,   7234.79425624200 ),
  { 4320   86 }  (         0.415, 0.96862624175,   4136.91043351620 ),
  { 4320   87 }  (         0.387, 3.09145061418,  25158.60171976540 ),
  { 4320   88 }  (         0.373, 2.65119262792,   7342.45778018060 ),
  { 4320   89 }  (         0.361, 2.97762937739,   9623.68827669120 ),
  { 4320   90 }  (         0.418, 3.75759994446,   5230.80746680300 ),
  { 4320   91 }  (         0.396, 1.22507712354,   6438.49624942560 ),
  { 4320   92 }  (         0.322, 1.21162178805,   8662.24032356300 ),
  { 4320   93 }  (         0.284, 5.64170320068,   1589.07289528380 ),
  { 4320   94 }  (         0.379, 1.72248432748,  14945.31617355440 ),
  { 4320   95 }  (         0.320, 3.94161159962,   7330.82316174610 ),
  { 4320   96 }  (         0.313, 5.47602376446,   1194.44701022460 ),
  { 4320   97 }  (         0.292, 1.38971327603,  11769.85369316640 ),
  { 4320   98 }  (         0.305, 0.80429352049,  37724.75341974820 ),
  { 4320   99 }  (         0.257, 5.81382809757,    426.59819087600 ),
  { 4320  100 }  (         0.265, 6.10358507671,   6836.64525283380 ),
  { 4320  101 }  (         0.250, 4.56452895547,   7477.52286021600 ),
  { 4320  102 }  (         0.266, 2.62926282354,   7238.67559160000 ),
  { 4320  103 }  (         0.263, 6.22089501237,   6133.51265285680 ),
  { 4320  104 }  (         0.306, 2.79682380531,   1748.01641306700 ),
  { 4320  105 }  (         0.236, 2.46093023714,  11371.70468975820 ),
  { 4320  106 }  (         0.316, 1.62662805006, 250908.49012041549 ),
  { 4320  107 }  (         0.216, 3.68721275185,   5849.36411211460 ),
  { 4320  108 }  (         0.230, 0.36165162947,   5863.59120611620 ),
  { 4320  109 }  (         0.233, 5.03509933858,  20426.57109242200 ),
  { 4320  110 }  (         0.200, 5.86073159059,   4535.05943692440 ),
  { 4320  111 }  (         0.277, 4.65400292395,  82239.16695779889 ),
  { 4320  112 }  (         0.209, 3.72323200804,  10973.55568635000 ),
  { 4320  113 }  (         0.199, 5.05186622555,   5429.87946823940 ),
  { 4320  114 }  (         0.256, 2.40923279770,  19651.04848109800 ),
  { 4320  115 }  (         0.210, 4.50691909144,  29088.81141598500 ),
  { 4320  116 }  (         0.181, 6.00294783127,   4292.33083295040 ),
  { 4320  117 }  (         0.249, 0.12900984422, 154379.79562448629 ),
  { 4320  118 }  (         0.209, 3.87759458598,  17789.84561978500 ),
  { 4320  119 }  (         0.225, 3.18339652605,  18875.52586977400 ),
  { 4320  120 }  (         0.191, 4.53897489299,  18477.10876461230 ),
  { 4320  121 }  (         0.172, 2.09694183014,  13095.84266507740 ),
  { 4320  122 }  (         0.182, 3.16107943500,  16730.46368959580 ),
  { 4320  123 }  (         0.188, 2.22746128596,  41654.96311596780 ),
  { 4320  124 }  (         0.164, 5.18686275017,   5481.25491886760 ),
  { 4320  125 }  (         0.160, 2.49298855159,  12592.45001978260 ),
  { 4320  126 }  (         0.155, 1.59595438230,  10021.83728009940 ),
  { 4320  127 }  (         0.135, 0.21349051064,  10988.80815753500 ),
  { 4320  128 }  (         0.178, 3.80375177970,  23581.25817731760 ),
  { 4320  129 }  (         0.123, 1.66800739151,  15110.46611986620 ),
  { 4320  130 }  (         0.122, 2.72678272244,  18849.22754997420 ),
  { 4320  131 }  (         0.126, 1.17675512910,  14919.01785375460 ),
  { 4320  132 }  (         0.142, 3.95053441332,    337.81426319640 ),
  { 4320  133 }  (         0.116, 6.06340906229,   6709.67404086740 ),
  { 4320  134 }  (         0.137, 3.52143246757,  12139.55350910680 ),
  { 4320  135 }  (         0.136, 2.92179113542,  32217.20018108080 ),
  { 4320  136 }  (         0.110, 3.51203379263,  18052.92954315780 ),
  { 4320  137 }  (         0.147, 4.63371971408,  22805.73556599360 ),
  { 4320  138 }  (         0.108, 5.45280814878,      7.11354700080 ),
  { 4320  139 }  (         0.148, 0.65447253687,  95480.94718417450 ),
  { 4320  140 }  (         0.119, 5.92110458985,  33019.02111220460 ),
  { 4320  141 }  (         0.110, 5.34824206306,    639.89728631400 ),
  { 4320  142 }  (         0.106, 3.71081682629,  14314.16811304980 ),
  { 4320  143 }  (         0.139, 6.17607198418,  24356.78078864160 ),
  { 4320  144 }  (         0.118, 5.59738712670, 161338.50000087050 ),
  { 4320  145 }  (         0.117, 3.65065271640,  45585.17281218740 ),
  { 4320  146 }  (         0.127, 4.74596574209,  49515.38250840700 ),
  { 4320  147 }  (         0.120, 1.04211499785,   6915.85958930460 ),
  { 4320  148 }  (         0.120, 5.60638811846,   5650.29211067820 ),
  { 4320  149 }  (         0.115, 3.10668213289,  14712.31711645800 ),
  { 4320  150 }  (         0.099, 0.69018940049,  12779.45079542080 ),
  { 4320  151 }  (         0.097, 1.07908724794,   9917.69687450980 ),
  { 4320  152 }  (         0.093, 2.62295197319,  17260.15465469040 ),
  { 4320  153 }  (         0.099, 4.45774681732,   4933.20844033260 ),
  { 4320  154 }  (         0.123, 1.37488922089,  28286.99048486120 ),
  { 4320  155 }  (         0.121, 5.19767249813,  27511.46787353720 ),
  { 4320  156 }  (         0.105, 0.87192267806,  77375.95720492408 ),
  { 4320  157 }  (         0.087, 3.93637812950,  17654.78053974960 ),
  { 4320  158 }  (         0.122, 2.23956068680,  83997.09113559539 ),
  { 4320  159 }  (         0.087, 4.18201600952,  22779.43724619380 ),
  { 4320  160 }  (         0.104, 4.59580877295,   1349.86740965880 ),
  { 4320  161 }  (         0.102, 2.83545248411,  12352.85260454480 ),
  { 4320  162 }  (         0.102, 3.97386522171,  10818.13528691580 ),
  { 4320  163 }  (         0.101, 4.32892825857,  36147.40987730040 ),
  { 4320  164 }  (         0.094, 5.00001709261, 150192.21439800429 ),
  { 4320  165 }  (         0.077, 3.97199369296,   1592.59601363280 ),
  { 4320  166 }  (         0.100, 6.07733097102,  26735.94526221320 ),
  { 4320  167 }  (         0.086, 5.26029638250,  28313.28880466100 ),
  { 4320  168 }  (         0.093, 4.31900620254,  44809.65020086340 ),
  { 4320  169 }  (         0.076, 6.22743405935,  13521.75144159140 ),
  { 4320  170 }  (         0.072, 1.55820597747,   6256.77753019160 ),
  { 4320  171 }  (         0.082, 4.95202664555,  10575.40668294180 ),
  { 4320  172 }  (         0.082, 1.69647647075,   1990.74501704100 ),
  { 4320  173 }  (         0.075, 2.29836095644,   3634.62102451840 ),
  { 4320  174 }  (         0.075, 2.66367876557,  16200.77272450120 ),
  { 4320  175 }  (         0.087, 0.26630214764,  31441.67756975680 ),
  { 4320  176 }  (         0.077, 2.25530954137,   5235.32853823670 ),
  { 4320  177 }  (         0.076, 1.09869730846,  12903.96596317920 ),
  { 4320  178 }  (         0.058, 4.28246138307,  12559.03815298200 ),
  { 4320  179 }  (         0.064, 5.51112830114, 173904.65170085328 ),
  { 4320  180 }  (         0.056, 2.60133794851,  73188.37597844210 ),
  { 4320  181 }  (         0.055, 5.81483150022, 143233.51002162008 ),
  { 4320  182 }  (         0.054, 3.38482031504, 323049.11878710288 ),
  { 4320  183 }  (         0.039, 3.28500401343,  71768.50988132549 ),
  { 4320  184 }  (         0.039, 3.11239910690,  96900.81328129109 )
  (*$endif *)
      );
  (*$ifdef meeus *)
  vsop87_ear_b1:array[0..  1,0..2] of extended = (
  (*$else *)
  vsop87_ear_b1:array[0.. 98,0..2] of extended = (
  (*$endif *)
  { 4321    1 }  (         9.030, 3.89729061890,   5507.55323866740 ),
  { 4321    2 }  (         6.177, 1.73038850355,   5223.69391980220 )
  (*$ifndef meeus *)                                                 ,
  { 4321    3 }  (         3.800, 5.24404145734,   2352.86615377180 ),
  { 4321    4 }  (         2.834, 2.47345037450,   1577.34354244780 ),
  { 4321    5 }  (         1.817, 0.41874743765,   6283.07584999140 ),
  { 4321    6 }  (         1.499, 1.83320979291,   5856.47765911540 ),
  { 4321    7 }  (         1.466, 5.69401926017,   5753.38488489680 ),
  { 4321    8 }  (         1.301, 2.18890066314,   9437.76293488700 ),
  { 4321    9 }  (         1.233, 4.95222451476,  10213.28554621100 ),
  { 4321   10 }  (         1.021, 0.12866660208,   7860.41939243920 ),
  { 4321   11 }  (         0.982, 0.09005453285,  14143.49524243060 ),
  { 4321   12 }  (         0.865, 1.73949953555,   3930.20969621960 ),
  { 4321   13 }  (         0.581, 2.26949174067,   5884.92684658320 ),
  { 4321   14 }  (         0.524, 5.65662503159,    529.69096509460 ),
  { 4321   15 }  (         0.473, 6.22750969242,   6309.37416979120 ),
  { 4321   16 }  (         0.451, 1.53288619213,  18073.70493865020 ),
  { 4321   17 }  (         0.364, 3.61614477374,  13367.97263110660 ),
  { 4321   18 }  (         0.372, 3.22470721320,   6275.96230299060 ),
  { 4321   19 }  (         0.268, 2.34341267879,  11790.62908865880 ),
  { 4321   20 }  (         0.322, 0.94084045832,   6069.77675455340 ),
  { 4321   21 }  (         0.232, 0.26781182579,   7058.59846131540 ),
  { 4321   22 }  (         0.216, 6.05952221329,  10977.07880469900 ),
  { 4321   23 }  (         0.232, 2.93325646109,  22003.91463486980 ),
  { 4321   24 }  (         0.204, 3.86264841382,   6496.37494542940 ),
  { 4321   25 }  (         0.202, 2.81892511133,  15720.83878487840 ),
  { 4321   26 }  (         0.185, 4.93512381859,  12036.46073488820 ),
  { 4321   27 }  (         0.220, 3.99305643742,   6812.76681508600 ),
  { 4321   28 }  (         0.166, 1.74970002999,  11506.76976979360 ),
  { 4321   29 }  (         0.212, 1.57166285369,   4694.00295470760 ),
  { 4321   30 }  (         0.157, 1.08259734788,   5643.17856367740 ),
  { 4321   31 }  (         0.154, 5.99434678412,   5486.77784317500 ),
  { 4321   32 }  (         0.144, 5.23285656085,  78051.58573131690 ),
  { 4321   33 }  (         0.144, 1.16454655948,  90617.73743129970 ),
  { 4321   34 }  (         0.137, 2.67760436027,   6290.18939699220 ),
  { 4321   35 }  (         0.180, 2.06509026215,   7084.89678111520 ),
  { 4321   36 }  (         0.121, 5.90212574947,   9225.53927328300 ),
  { 4321   37 }  (         0.150, 2.00175038718,   5230.80746680300 ),
  { 4321   38 }  (         0.149, 5.06157254516,  17298.18232732620 ),
  { 4321   39 }  (         0.118, 5.39979058038,   3340.61242669980 ),
  { 4321   40 }  (         0.161, 3.32421999691,   6283.31966747490 ),
  { 4321   41 }  (         0.121, 4.36722193162,  19651.04848109800 ),
  { 4321   42 }  (         0.116, 5.83462858507,   4705.73230754360 ),
  { 4321   43 }  (         0.128, 4.35489873365,  25934.12433108940 ),
  { 4321   44 }  (         0.143, 0.00000000000,      0.00000000000 ),
  { 4321   45 }  (         0.109, 2.52157834166,   6438.49624942560 ),
  { 4321   46 }  (         0.099, 2.70727488041,   5216.58037280140 ),
  { 4321   47 }  (         0.103, 0.93782340879,   8827.39026987480 ),
  { 4321   48 }  (         0.082, 4.29214680390,   8635.94200376320 ),
  { 4321   49 }  (         0.079, 2.24085737326,   1059.38193018920 ),
  { 4321   50 }  (         0.097, 5.50959692365,  29864.33402730900 ),
  { 4321   51 }  (         0.072, 0.21891639822,  21228.39202354580 ),
  { 4321   52 }  (         0.071, 2.86755026812,   6681.22485339960 ),
  { 4321   53 }  (         0.074, 2.20184828895,  37724.75341974820 ),
  { 4321   54 }  (         0.063, 4.45586625948,   7079.37385680780 ),
  { 4321   55 }  (         0.061, 0.63918772258,  33794.54372352860 ),
  { 4321   56 }  (         0.047, 2.09070235724,   3128.38876509580 ),
  { 4321   57 }  (         0.047, 3.32543843300,  26087.90314157420 ),
  { 4321   58 }  (         0.049, 1.60680905005,   6702.56049386660 ),
  { 4321   59 }  (         0.057, 0.11215813438,  29088.81141598500 ),
  { 4321   60 }  (         0.056, 5.47982934911,    775.52261132400 ),
  { 4321   61 }  (         0.050, 1.89396788463,  12139.55350910680 ),
  { 4321   62 }  (         0.047, 2.97214907240,  20426.57109242200 ),
  { 4321   63 }  (         0.041, 5.55329394890,  11015.10647733480 ),
  { 4321   64 }  (         0.041, 5.91861144924,  23581.25817731760 ),
  { 4321   65 }  (         0.045, 4.95273290181,   5863.59120611620 ),
  { 4321   66 }  (         0.050, 3.62740835096,  41654.96311596780 ),
  { 4321   67 }  (         0.037, 6.09033460601,  64809.80550494129 ),
  { 4321   68 }  (         0.037, 5.86153655431,  12566.15169998280 ),
  { 4321   69 }  (         0.046, 1.65798680284,  25158.60171976540 ),
  { 4321   70 }  (         0.038, 2.00673650251,    426.59819087600 ),
  { 4321   71 }  (         0.036, 6.24373396652,   6283.14316029419 ),
  { 4321   72 }  (         0.036, 0.40465162918,   6283.00853968860 ),
  { 4321   73 }  (         0.032, 6.03707103538,   2942.46342329160 ),
  { 4321   74 }  (         0.041, 4.86809570283,   1592.59601363280 ),
  { 4321   75 }  (         0.028, 4.38359423735,   7632.94325965020 ),
  { 4321   76 }  (         0.028, 6.03334294232,  17789.84561978500 ),
  { 4321   77 }  (         0.026, 3.88971333608,   5331.35744374080 ),
  { 4321   78 }  (         0.026, 5.94932724051,  16496.36139620240 ),
  { 4321   79 }  (         0.031, 1.44666331503,  16730.46368959580 ),
  { 4321   80 }  (         0.026, 6.26376705837,  23543.23050468179 ),
  { 4321   81 }  (         0.033, 0.93797239147,    213.29909543800 ),
  { 4321   82 }  (         0.026, 3.71858432944,  13095.84266507740 ),
  { 4321   83 }  (         0.027, 0.60565274405,  10988.80815753500 ),
  { 4321   84 }  (         0.023, 4.44388985550,  18849.22754997420 ),
  { 4321   85 }  (         0.028, 1.53862289477,   6279.48542133960 ),
  { 4321   86 }  (         0.028, 1.96831814872,   6286.66627864320 ),
  { 4321   87 }  (         0.028, 5.78094918529,  15110.46611986620 ),
  { 4321   88 }  (         0.026, 2.48165809843,   5729.50644714900 ),
  { 4321   89 }  (         0.020, 3.85655029499,   9623.68827669120 ),
  { 4321   90 }  (         0.021, 5.83006047147,   7234.79425624200 ),
  { 4321   91 }  (         0.021, 0.69628570421,    398.14900340820 ),
  { 4321   92 }  (         0.022, 5.02222806555,   6127.65545055720 ),
  { 4321   93 }  (         0.020, 3.47611265290,   6148.01076995600 ),
  { 4321   94 }  (         0.020, 0.90769829044,   5481.25491886760 ),
  { 4321   95 }  (         0.020, 0.03081589303,   6418.14093002680 ),
  { 4321   96 }  (         0.020, 3.74220084927,   1589.07289528380 ),
  { 4321   97 }  (         0.021, 4.00149269576,   3154.68708489560 ),
  { 4321   98 }  (         0.018, 1.58348238359,   2118.76386037840 ),
  { 4321   99 }  (         0.019, 0.85407021371,  14712.31711645800 )
  (*$endif *)
      );
  (*$ifdef meeus *)
  vsop87_ear_b2:array[0..  0,0..2] of extended = (
  (*$else *)
  vsop87_ear_b2:array[0.. 48,0..2] of extended = (
  (*$endif *)
  (*$ifdef meeus *)
                 (         0.000, 0.00000000000,      0.00000000000 )
  (*$else *)
  { 4322    1 }  (         1.662, 1.62703209173,  84334.66158130829 ),
  { 4322    2 }  (         0.492, 2.41382223971,   1047.74731175470 ),
  { 4322    3 }  (         0.344, 2.24353004539,   5507.55323866740 ),
  { 4322    4 }  (         0.258, 6.00906896311,   5223.69391980220 ),
  { 4322    5 }  (         0.131, 0.95447345240,   6283.07584999140 ),
  { 4322    6 }  (         0.086, 1.67530247303,   7860.41939243920 ),
  { 4322    7 }  (         0.090, 0.97606804452,   1577.34354244780 ),
  { 4322    8 }  (         0.090, 0.37899871725,   2352.86615377180 ),
  { 4322    9 }  (         0.089, 6.25807507963,  10213.28554621100 ),
  { 4322   10 }  (         0.075, 0.84213523741, 167621.57585086189 ),
  { 4322   11 }  (         0.052, 1.70501566089,  14143.49524243060 ),
  { 4322   12 }  (         0.057, 6.15295833679,  12194.03291462090 ),
  { 4322   13 }  (         0.051, 1.27616016740,   5753.38488489680 ),
  { 4322   14 }  (         0.051, 5.37229738682,   6812.76681508600 ),
  { 4322   15 }  (         0.034, 1.73672994279,   7058.59846131540 ),
  { 4322   16 }  (         0.038, 2.77761031485,  10988.80815753500 ),
  { 4322   17 }  (         0.046, 3.38617099014, 156475.29024799570 ),
  { 4322   18 }  (         0.021, 1.95248349228,   8827.39026987480 ),
  { 4322   19 }  (         0.018, 3.33419222028,   8429.24126646660 ),
  { 4322   20 }  (         0.019, 4.32945160287,  17789.84561978500 ),
  { 4322   21 }  (         0.017, 0.66191210656,   6283.00853968860 ),
  { 4322   22 }  (         0.018, 3.74885333072,  11769.85369316640 ),
  { 4322   23 }  (         0.017, 4.23058370776,  10977.07880469900 ),
  { 4322   24 }  (         0.017, 1.78116162721,   5486.77784317500 ),
  { 4322   25 }  (         0.021, 1.36972913918,  12036.46073488820 ),
  { 4322   26 }  (         0.017, 2.79601092529,    796.29800681640 ),
  { 4322   27 }  (         0.015, 0.43087848850,  11790.62908865880 ),
  { 4322   28 }  (         0.017, 1.35132152761,  78051.58573131690 ),
  { 4322   29 }  (         0.015, 1.17032155085,    213.29909543800 ),
  { 4322   30 }  (         0.018, 2.85221514199,   5088.62883976680 ),
  { 4322   31 }  (         0.017, 0.21780913672,   6283.14316029419 ),
  { 4322   32 }  (         0.013, 1.21201504386,  25132.30339996560 ),
  { 4322   33 }  (         0.012, 1.12953712197,  90617.73743129970 ),
  { 4322   34 }  (         0.012, 5.13714452592,   7079.37385680780 ),
  { 4322   35 }  (         0.013, 3.79842135217,   4933.20844033260 ),
  { 4322   36 }  (         0.012, 4.89407978213,   3738.76143010800 ),
  { 4322   37 }  (         0.015, 6.05682328852,    398.14900340820 ),
  { 4322   38 }  (         0.014, 4.81029291856,   4694.00295470760 ),
  { 4322   39 }  (         0.011, 0.61684523405,   3128.38876509580 ),
  { 4322   40 }  (         0.011, 5.32876538500,   6040.34724601740 ),
  { 4322   41 }  (         0.014, 5.27227350286,   4535.05943692440 ),
  { 4322   42 }  (         0.011, 2.39292099451,   5331.35744374080 ),
  { 4322   43 }  (         0.010, 4.45296532710,   6525.80445396540 ),
  { 4322   44 }  (         0.014, 4.66400985037,   8031.09226305840 ),
  { 4322   45 }  (         0.010, 3.22472385926,   9437.76293488700 ),
  { 4322   46 }  (         0.011, 3.80913404437,    801.82093112380 ),
  { 4322   47 }  (         0.010, 5.15032130575,  11371.70468975820 ),
  { 4322   48 }  (         0.013, 0.98720797401,   5729.50644714900 ),
  { 4322   49 }  (         0.009, 5.94191743597,   7632.94325965020 )
  (*$endif *)
      );
  (*$ifdef meeus *)
  vsop87_ear_b3:array[0..  0,0..2] of extended = (
  (*$else *)
  vsop87_ear_b3:array[0.. 10,0..2] of extended = (
  (*$endif *)
  (*$ifdef meeus *)
                 (         0.000, 0.00000000000,      0.00000000000 )
  (*$else *)
  { 4323    1 }  (         0.011, 0.23877262399,   7860.41939243920 ),
  { 4323    2 }  (         0.009, 1.16069982609,   5507.55323866740 ),
  { 4323    3 }  (         0.008, 1.65357552925,   5884.92684658320 ),
  { 4323    4 }  (         0.008, 2.86720038197,   7058.59846131540 ),
  { 4323    5 }  (         0.007, 3.04818741666,   5486.77784317500 ),
  { 4323    6 }  (         0.007, 2.59437103785,    529.69096509460 ),
  { 4323    7 }  (         0.008, 4.02863090524,   6256.77753019160 ),
  { 4323    8 }  (         0.008, 2.42003508927,   5753.38488489680 ),
  { 4323    9 }  (         0.006, 0.84181087594,   6275.96230299060 ),
  { 4323   10 }  (         0.006, 5.40160929468,   1577.34354244780 ),
  { 4323   11 }  (         0.007, 2.73399865247,   6309.37416979120 )
  (*$endif *)
      );
  (*$ifdef meeus *)
  vsop87_ear_b4:array[0..  0,0..2] of extended = (
  (*$else *)
  vsop87_ear_b4:array[0..  4,0..2] of extended = (
  (*$endif *)
  (*$ifdef meeus *)
                 (         0.000, 0.00000000000,      0.00000000000 )
  (*$else *)
  { 4324    1 }  ( 000000000.004, 0.79662198849,   6438.49624942560 ),
  { 4324    2 }  ( 000000000.005, 0.84308705203,   1047.74731175470 ),
  { 4324    3 }  ( 000000000.005, 0.05711572303,  84334.66158130829 ),
  { 4324    4 }  ( 000000000.003, 3.46779895686,   6279.55273164240 ),
  { 4324    5 }  ( 000000000.003, 2.89822201212,   6127.65545055720 )
  (*$endif *)
      );
begin
  WITH result do begin
    a:=0; b:=0; c:=0;
    case index of
      0: if (nr>=low(vsop87_ear_b0)) and (nr<=high(vsop87_ear_b0)) then begin
        a:=vsop87_ear_b0[nr,0];  b:=vsop87_ear_b0[nr,1];  c:=vsop87_ear_b0[nr,2];
        end;
      1: if (nr>=low(vsop87_ear_b1)) and (nr<=high(vsop87_ear_b1)) then begin
        a:=vsop87_ear_b1[nr,0];  b:=vsop87_ear_b1[nr,1];  c:=vsop87_ear_b1[nr,2];
        end;
      2: if (nr>=low(vsop87_ear_b2)) and (nr<=high(vsop87_ear_b2)) then begin
        a:=vsop87_ear_b2[nr,0];  b:=vsop87_ear_b2[nr,1];  c:=vsop87_ear_b2[nr,2];
        end;
      3: if (nr>=low(vsop87_ear_b3)) and (nr<=high(vsop87_ear_b3)) then begin
        a:=vsop87_ear_b3[nr,0];  b:=vsop87_ear_b3[nr,1];  c:=vsop87_ear_b3[nr,2];
        end;
      4: if (nr>=low(vsop87_ear_b4)) and (nr<=high(vsop87_ear_b4)) then begin
        a:=vsop87_ear_b4[nr,0];  b:=vsop87_ear_b4[nr,1];  c:=vsop87_ear_b4[nr,2];
        end;
      end;
    end;
  end;

function TVSOPEarth.LongitudeFactor(nr,index: integer):TVSOPEntry;
const
  (*$ifdef meeus *)
  vsop87_ear_l0:array[0.. 63,0..2] of extended = (
  (*$else *)
  vsop87_ear_l0:array[0..558,0..2] of extended = (
  (*$endif *)
  { 4310    1 }  ( 175347045.673, 0.00000000000,      0.00000000000 ),
  { 4310    2 }  (   3341656.456, 4.66925680417,   6283.07584999140 ),
  { 4310    3 }  (     34894.275, 4.62610241759,  12566.15169998280 ),
  { 4310    4 }  (      3417.571, 2.82886579606,      3.52311834900 ),
  { 4310    5 }  (      3497.056, 2.74411800971,   5753.38488489680 ),
  { 4310    6 }  (      3135.896, 3.62767041758,  77713.77146812050 ),
  { 4310    7 }  (      2676.218, 4.41808351397,   7860.41939243920 ),
  { 4310    8 }  (      2342.687, 6.13516237631,   3930.20969621960 ),
  { 4310    9 }  (      1273.166, 2.03709655772,    529.69096509460 ),
  { 4310   10 }  (      1324.292, 0.74246356352,  11506.76976979360 ),
  { 4310   11 }  (       901.855, 2.04505443513,     26.29831979980 ),
  { 4310   12 }  (      1199.167, 1.10962944315,   1577.34354244780 ),
  { 4310   13 }  (       857.223, 3.50849156957,    398.14900340820 ),
  { 4310   14 }  (       779.786, 1.17882652114,   5223.69391980220 ),
  { 4310   15 }  (       990.250, 5.23268129594,   5884.92684658320 ),
  { 4310   16 }  (       753.141, 2.53339053818,   5507.55323866740 ),
  { 4310   17 }  (       505.264, 4.58292563052,  18849.22754997420 ),
  { 4310   18 }  (       492.379, 4.20506639861,    775.52261132400 ),
  { 4310   19 }  (       356.655, 2.91954116867,      0.06731030280 ),
  { 4310   20 }  (       284.125, 1.89869034186,    796.29800681640 ),
  { 4310   21 }  (       242.810, 0.34481140906,   5486.77784317500 ),
  { 4310   22 }  (       317.087, 5.84901952218,  11790.62908865880 ),
  { 4310   23 }  (       271.039, 0.31488607649,  10977.07880469900 ),
  { 4310   24 }  (       206.160, 4.80646606059,   2544.31441988340 ),
  { 4310   25 }  (       205.385, 1.86947813692,   5573.14280143310 ),
  { 4310   26 }  (       202.261, 2.45767795458,   6069.77675455340 ),
  { 4310   27 }  (       126.184, 1.08302630210,     20.77539549240 ),
  { 4310   28 }  (       155.516, 0.83306073807,    213.29909543800 ),
  { 4310   29 }  (       115.132, 0.64544911683,      0.98032106820 ),
  { 4310   30 }  (       102.851, 0.63599846727,   4694.00295470760 ),
  { 4310   31 }  (       101.724, 4.26679821365,      7.11354700080 ),
  { 4310   32 }  (        99.206, 6.20992940258,   2146.16541647520 ),
  { 4310   33 }  (       132.212, 3.41118275555,   2942.46342329160 ),
  { 4310   34 }  (        97.607, 0.68101272270,    155.42039943420 ),
  { 4310   35 }  (        85.128, 1.29870743025,   6275.96230299060 ),
  { 4310   36 }  (        74.651, 1.75508916159,   5088.62883976680 ),
  { 4310   37 }  (       101.895, 0.97569221824,  15720.83878487840 ),
  { 4310   38 }  (        84.711, 3.67080093025,  71430.69561812909 ),
  { 4310   39 }  (        73.547, 4.67926565481,    801.82093112380 ),
  { 4310   40 }  (        73.874, 3.50319443167,   3154.68708489560 ),
  { 4310   41 }  (        78.756, 3.03698313141,  12036.46073488820 ),
  { 4310   42 }  (        79.637, 1.80791330700,  17260.15465469040 ),
  { 4310   43 }  (        85.803, 5.98322631256, 161000.68573767410 ),
  { 4310   44 }  (        56.963, 2.78430398043,   6286.59896834040 ),
  { 4310   45 }  (        61.148, 1.81839811024,   7084.89678111520 ),
  { 4310   46 }  (        69.627, 0.83297596966,   9437.76293488700 ),
  { 4310   47 }  (        56.116, 4.38694880779,  14143.49524243060 ),
  { 4310   48 }  (        62.449, 3.97763880587,   8827.39026987480 ),
  { 4310   49 }  (        51.145, 0.28306864501,   5856.47765911540 ),
  { 4310   50 }  (        55.577, 3.47006009062,   6279.55273164240 ),
  { 4310   51 }  (        41.036, 5.36817351402,   8429.24126646660 ),
  { 4310   52 }  (        51.605, 1.33282746983,   1748.01641306700 ),
  { 4310   53 }  (        51.992, 0.18914945834,  12139.55350910680 ),
  { 4310   54 }  (        49.000, 0.48735065033,   1194.44701022460 ),
  { 4310   55 }  (        39.200, 6.16832995016,  10447.38783960440 ),
  { 4310   56 }  (        35.566, 1.77597314691,   6812.76681508600 ),
  { 4310   57 }  (        36.770, 6.04133859347,  10213.28554621100 ),
  { 4310   58 }  (        36.596, 2.56955238628,   1059.38193018920 ),
  { 4310   59 }  (        33.291, 0.59309499459,  17789.84561978500 ),
  { 4310   60 }  (        35.954, 1.70876111898,   2352.86615377180 ),
  { 4310   61 }  (        40.938, 2.39850881707,  19651.04848109800 ),
  { 4310   62 }  (        30.047, 2.73975123935,   1349.86740965880 ),
  { 4310   63 }  (        30.412, 0.44294464135,  83996.84731811189 ),
  (*$ifndef meeus *)
  { 4310   64 }  (        23.663, 0.48473567763,   8031.09226305840 ),
  { 4310   65 }  (        23.574, 2.06527720049,   3340.61242669980 ),
  { 4310   66 }  (        21.089, 4.14825464101,    951.71840625060 ),
  { 4310   67 }  (        24.738, 0.21484762138,      3.59042865180 ),
  (*$endif *)
  { 4310   68 }  (        25.352, 3.16470953405,   4690.47983635860 )
  (*$ifndef meeus *)                                                 ,
  { 4310   69 }  (        22.820, 5.22197888032,   4705.73230754360 ),
  { 4310   70 }  (        21.419, 1.42563735525,  16730.46368959580 ),
  { 4310   71 }  (        21.891, 5.55594302562,    553.56940284240 ),
  { 4310   72 }  (        17.481, 4.56052900359,    135.06508003540 ),
  { 4310   73 }  (        19.925, 5.22208471269,  12168.00269657460 ),
  { 4310   74 }  (        19.860, 5.77470167653,   6309.37416979120 ),
  { 4310   75 }  (        20.300, 0.37133792946,    283.85931886520 ),
  { 4310   76 }  (        14.421, 4.19315332546,    242.72860397400 ),
  { 4310   77 }  (        16.225, 5.98837722564,  11769.85369316640 ),
  { 4310   78 }  (        15.077, 4.19567181073,   6256.77753019160 ),
  { 4310   79 }  (        19.124, 3.82219996949,  23581.25817731760 ),
  { 4310   80 }  (        18.888, 5.38626880969, 149854.40013480789 ),
  { 4310   81 }  (        14.346, 3.72355084422,     38.02767263580 ),
  { 4310   82 }  (        17.898, 2.21490735647,  13367.97263110660 ),
  { 4310   83 }  (        12.054, 2.62229588349,    955.59974160860 ),
  { 4310   84 }  (        11.287, 0.17739328092,   4164.31198961300 ),
  { 4310   85 }  (        13.971, 4.40138139996,   6681.22485339960 ),
  { 4310   86 }  (        13.621, 1.88934471407,   7632.94325965020 ),
  { 4310   87 }  (        12.503, 1.13052412208,      5.52292430740 ),
  { 4310   88 }  (        10.498, 5.35909518669,   1592.59601363280 ),
  { 4310   89 }  (         9.803, 0.99947478995,  11371.70468975820 ),
  { 4310   90 }  (         9.220, 4.57138609781,   4292.33083295040 ),
  { 4310   91 }  (        10.327, 6.19982566125,   6438.49624942560 ),
  { 4310   92 }  (        12.003, 1.00351456700,    632.78373931320 ),
  { 4310   93 }  (        10.827, 0.32734520222,    103.09277421860 ),
  { 4310   94 }  (         8.356, 4.53902685948,  25132.30339996560 ),
  { 4310   95 }  (        10.005, 6.02914963280,   5746.27133789600 ),
  { 4310   96 }  (         8.409, 3.29946744189,   7234.79425624200 ),
  { 4310   97 }  (         8.006, 5.82145271907,     28.44918746780 ),
  { 4310   98 }  (        10.523, 0.93871805506,  11926.25441366880 ),
  { 4310   99 }  (         7.686, 3.12142363172,   7238.67559160000 ),
  { 4310  100 }  (         9.378, 2.62414241032,   5760.49843189760 ),
  { 4310  101 }  (         8.127, 6.11228001785,   4732.03062734340 ),
  { 4310  102 }  (         9.232, 0.48343968736,    522.57741809380 ),
  { 4310  103 }  (         9.802, 5.24413991147,  27511.46787353720 ),
  { 4310  104 }  (         7.871, 0.99590177926,   5643.17856367740 ),
  { 4310  105 }  (         8.123, 6.27053013650,    426.59819087600 ),
  { 4310  106 }  (         9.048, 5.33686335897,   6386.16862421000 ),
  { 4310  107 }  (         8.620, 4.16538210888,   7058.59846131540 ),
  { 4310  108 }  (         6.297, 4.71724819317,   6836.64525283380 ),
  { 4310  109 }  (         7.575, 3.97382858911,  11499.65622279280 ),
  { 4310  110 }  (         7.756, 2.95729056763,  23013.53953958720 ),
  { 4310  111 }  (         7.314, 0.60652505806,  11513.88331679440 ),
  { 4310  112 }  (         5.955, 2.87641047971,   6283.14316029419 ),
  { 4310  113 }  (         6.534, 5.79072926033,  18073.70493865020 ),
  { 4310  114 }  (         7.188, 3.99831508699,     74.78159856730 ),
  { 4310  115 }  (         7.346, 4.38582365437,    316.39186965660 ),
  { 4310  116 }  (         5.413, 5.39199024641,    419.48464387520 ),
  { 4310  117 }  (         5.127, 2.36062848786,  10973.55568635000 ),
  { 4310  118 }  (         7.056, 0.32258441903,    263.08392337280 ),
  { 4310  119 }  (         6.625, 3.66475158672,  17298.18232732620 ),
  { 4310  120 }  (         6.762, 5.91132535899,  90955.55169449610 ),
  { 4310  121 }  (         4.938, 5.73672165674,   9917.69687450980 ),
  { 4310  122 }  (         5.547, 2.45152597661,  12352.85260454480 ),
  { 4310  123 }  (         5.958, 3.32051344676,   6283.00853968860 ),
  { 4310  124 }  (         4.471, 2.06385999536,   7079.37385680780 ),
  { 4310  125 }  (         6.153, 1.45823331144, 233141.31440436149 ),
  { 4310  126 }  (         4.348, 4.42342175480,   5216.58037280140 ),
  { 4310  127 }  (         6.123, 1.07494905258,  19804.82729158280 ),
  { 4310  128 }  (         4.488, 3.65285037150,    206.18554843720 ),
  { 4310  129 }  (         4.020, 0.83995823171,     20.35531939880 ),
  { 4310  130 }  (         5.188, 4.06503864016,   6208.29425142410 ),
  { 4310  131 }  (         5.307, 0.38217636096,  31441.67756975680 ),
  { 4310  132 }  (         3.785, 2.34369213733,      3.88133535800 ),
  { 4310  133 }  (         4.497, 3.27230796845,  11015.10647733480 ),
  { 4310  134 }  (         4.132, 0.92128915753,   3738.76143010800 ),
  { 4310  135 }  (         3.521, 5.97844807108,   3894.18182954220 ),
  { 4310  136 }  (         4.215, 1.90601120623,    245.83164622940 ),
  { 4310  137 }  (         3.701, 5.03069397926,    536.80451209540 ),
  { 4310  138 }  (         3.865, 1.82634360607,  11856.21865142450 ),
  { 4310  139 }  (         3.652, 1.01838584934,  16200.77272450120 ),
  { 4310  140 }  (         3.390, 0.97785123922,   8635.94200376320 ),
  { 4310  141 }  (         3.737, 2.95380107829,   3128.38876509580 ),
  { 4310  142 }  (         3.507, 3.71291946325,   6290.18939699220 ),
  { 4310  143 }  (         3.086, 3.64646921512,     10.63666534980 ),
  { 4310  144 }  (         3.397, 1.10590684017,  14712.31711645800 ),
  { 4310  145 }  (         3.334, 0.83684924911,   6496.37494542940 ),
  { 4310  146 }  (         2.805, 2.58504514144,  14314.16811304980 ),
  { 4310  147 }  (         3.650, 1.08344142571,  88860.05707098669 ),
  { 4310  148 }  (         3.388, 3.20185096055,   5120.60114558360 ),
  { 4310  149 }  (         3.252, 3.47859752062,   6133.51265285680 ),
  { 4310  150 }  (         2.553, 3.94869034189,   1990.74501704100 ),
  { 4310  151 }  (         3.520, 2.05559692878, 244287.60000722769 ),
  { 4310  152 }  (         2.565, 1.56071784900,  23543.23050468179 ),
  { 4310  153 }  (         2.621, 3.85639359951,    266.60704172180 ),
  { 4310  154 }  (         2.955, 3.39692949667,   9225.53927328300 ),
  { 4310  155 }  (         2.876, 6.02635617464, 154717.60988768269 ),
  { 4310  156 }  (         2.395, 1.16131956403,  10984.19235169980 ),
  { 4310  157 }  (         3.161, 1.32798718453,  10873.98603048040 ),
  { 4310  158 }  (         3.163, 5.08946464629,  21228.39202354580 ),
  { 4310  159 }  (         2.361, 4.27212906992,   6040.34724601740 ),
  { 4310  160 }  (         3.030, 1.80209931347,  35371.88726597640 ),
  { 4310  161 }  (         2.343, 3.57689860500,  10969.96525769820 ),
  { 4310  162 }  (         2.618, 2.57870156528,  22483.84857449259 ),
  { 4310  163 }  (         2.113, 3.71393780256,  65147.61976813770 ),
  { 4310  164 }  (         2.019, 0.81393923319,    170.67287061920 ),
  { 4310  165 }  (         2.003, 0.38091017375,   6172.86952877200 ),
  { 4310  166 }  (         2.506, 3.74379142438,  10575.40668294180 ),
  { 4310  167 }  (         2.381, 0.10581361289,      7.04623669800 ),
  { 4310  168 }  (         1.949, 4.86892513469,     36.02786667740 ),
  { 4310  169 }  (         2.074, 4.22794774570,   5650.29211067820 ),
  { 4310  170 }  (         1.924, 5.59460549860,   6282.09552892320 ),
  { 4310  171 }  (         1.949, 1.07002512703,   5230.80746680300 ),
  { 4310  172 }  (         1.988, 5.19736046771,   6262.30045449900 ),
  { 4310  173 }  (         1.887, 3.74365662683,     23.87843774780 ),
  { 4310  174 }  (         1.787, 1.25929682929,  12559.03815298200 ),
  { 4310  175 }  (         1.883, 1.90364058477,     15.25247118500 ),
  { 4310  176 }  (         1.816, 3.68083868442,  15110.46611986620 ),
  { 4310  177 }  (         1.701, 4.41105895380,    110.20632121940 ),
  { 4310  178 }  (         1.990, 3.93295788548,   6206.80977871580 ),
  { 4310  179 }  (         2.103, 0.75354917468,  13521.75144159140 ),
  { 4310  180 }  (         1.774, 0.48747535361,   1551.04522264800 ),
  { 4310  181 }  (         1.882, 0.86684493432,  22003.91463486980 ),
  { 4310  182 }  (         1.924, 1.22898324132,    709.93304855830 ),
  { 4310  183 }  (         2.009, 4.62850921980,   6037.24420376200 ),
  { 4310  184 }  (         1.924, 0.60231842508,   6284.05617105960 ),
  { 4310  185 }  (         1.596, 3.98332956992,  13916.01910964160 ),
  { 4310  186 }  (         1.664, 4.41939715469,   8662.24032356300 ),
  { 4310  187 }  (         1.971, 1.04560500503,  18209.33026366019 ),
  { 4310  188 }  (         1.942, 4.31335979989,   6244.94281435360 ),
  { 4310  189 }  (         1.476, 0.93271367331,   2379.16447357160 ),
  { 4310  190 }  (         1.810, 0.49112137707,      1.48447270830 ),
  { 4310  191 }  (         1.346, 1.51574702235,   4136.91043351620 ),
  { 4310  192 }  (         1.528, 5.61835711404,   6127.65545055720 ),
  { 4310  193 }  (         1.791, 3.22187270126,  39302.09696219600 ),
  { 4310  194 }  (         1.747, 3.05638656738,  18319.53658487960 ),
  { 4310  195 }  (         1.431, 4.51153808594,  20426.57109242200 ),
  { 4310  196 }  (         1.695, 0.22047718414,  25158.60171976540 ),
  { 4310  197 }  (         1.242, 4.46665769933,  17256.63153634140 ),
  { 4310  198 }  (         1.463, 4.69242679213,  14945.31617355440 ),
  { 4310  199 }  (         1.205, 1.86912144659,   4590.91018048900 ),
  { 4310  200 }  (         1.192, 2.74227166898,  12569.67481833180 ),
  { 4310  201 }  (         1.222, 5.18120087482,   5333.90024102160 ),
  { 4310  202 }  (         1.390, 5.42894648983, 143571.32428481648 ),
  { 4310  203 }  (         1.473, 1.70479245805,  11712.95531823080 ),
  { 4310  204 }  (         1.362, 2.61069503292,   6062.66320755260 ),
  { 4310  205 }  (         1.148, 6.03001800540,   3634.62102451840 ),
  { 4310  206 }  (         1.198, 5.15294130422,  10177.25767953360 ),
  { 4310  207 }  (         1.266, 0.11421493643,  18422.62935909819 ),
  { 4310  208 }  (         1.411, 1.09908857534,   3496.03282613400 ),
  { 4310  209 }  (         1.349, 2.99805109633,  17654.78053974960 ),
  { 4310  210 }  (         1.253, 2.79850152848, 167283.76158766549 ),
  { 4310  211 }  (         1.311, 1.60942984879,   5481.25491886760 ),
  { 4310  212 }  (         1.079, 6.20304501787,      3.28635741780 ),
  { 4310  213 }  (         1.181, 1.20653776978,    131.54196168640 ),
  { 4310  214 }  (         1.254, 5.45103277798,   6076.89030155420 ),
  { 4310  215 }  (         1.035, 2.32142722747,   7342.45778018060 ),
  { 4310  216 }  (         1.117, 0.38838354256,    949.17560896980 ),
  { 4310  217 }  (         0.966, 3.18341890851,  11087.28512591840 ),
  { 4310  218 }  (         1.171, 3.39635049962,  12562.62858163380 ),
  { 4310  219 }  (         1.121, 0.72627490378,    220.41264243880 ),
  { 4310  220 }  (         1.024, 2.19378315386,  11403.67699557500 ),
  { 4310  221 }  (         0.888, 3.91173199285,   4686.88940770680 ),
  { 4310  222 }  (         0.910, 1.98802695087,    735.87651353180 ),
  { 4310  223 }  (         0.830, 0.48984915507,  24072.92146977640 ),
  { 4310  224 }  (         1.096, 6.17377835617,   5436.99301524020 ),
  { 4310  225 }  (         0.908, 0.44959639433,   7477.52286021600 ),
  { 4310  226 }  (         0.974, 1.52996238356,   9623.68827669120 ),
  { 4310  227 }  (         0.840, 1.79543266333,   5429.87946823940 ),
  { 4310  228 }  (         0.778, 6.17699177946,     38.13303563780 ),
  { 4310  229 }  (         0.776, 4.09855402433,     14.22709400160 ),
  { 4310  230 }  (         1.068, 4.64200173735,  43232.30665841560 ),
  { 4310  231 }  (         0.954, 1.49988435748,   1162.47470440780 ),
  { 4310  232 }  (         0.907, 0.86986870809,  10344.29506538580 ),
  { 4310  233 }  (         0.931, 4.06044689031,  28766.92442448400 ),
  { 4310  234 }  (         0.739, 5.04368197372,    639.89728631400 ),
  { 4310  235 }  (         0.937, 3.46884698960,   1589.07289528380 ),
  { 4310  236 }  (         0.763, 5.86304932998,  16858.48253293320 ),
  { 4310  237 }  (         0.953, 4.20801492835,  11190.37790013700 ),
  { 4310  238 }  (         0.708, 1.72899988940,  13095.84266507740 ),
  { 4310  239 }  (         0.969, 1.64439522215,  29088.81141598500 ),
  { 4310  240 }  (         0.717, 0.16688678895,     11.72935283600 ),
  { 4310  241 }  (         0.962, 3.53092337542,  12416.58850284820 ),
  { 4310  242 }  (         0.747, 5.77866940346,  12592.45001978260 ),
  { 4310  243 }  (         0.672, 1.91095796194,      3.93215326310 ),
  { 4310  244 }  (         0.671, 5.46240843677,  18052.92954315780 ),
  { 4310  245 }  (         0.675, 6.28311558823,   4535.05943692440 ),
  { 4310  246 }  (         0.684, 0.39975012080,   5849.36411211460 ),
  { 4310  247 }  (         0.799, 0.29851185294,  12132.43996210600 ),
  { 4310  248 }  (         0.758, 0.96370823331,   1052.26838318840 ),
  { 4310  249 }  (         0.782, 5.33878339919,  13517.87010623340 ),
  { 4310  250 }  (         0.730, 1.70106160291,  17267.26820169119 ),
  { 4310  251 }  (         0.749, 2.59599901875,  11609.86254401220 ),
  { 4310  252 }  (         0.734, 2.78417782952,    640.87760738220 ),
  { 4310  253 }  (         0.688, 5.15048287468,  16496.36139620240 ),
  { 4310  254 }  (         0.770, 1.62469589333,   4701.11650170840 ),
  { 4310  255 }  (         0.633, 2.20587893893,  25934.12433108940 ),
  { 4310  256 }  (         0.760, 4.21317219403,    377.37360791580 ),
  { 4310  257 }  (         0.584, 2.13420121623,  10557.59416082380 ),
  { 4310  258 }  (         0.574, 0.24250054587,   9779.10867612540 ),
  { 4310  259 }  (         0.573, 3.16435264609,    533.21408344360 ),
  { 4310  260 }  (         0.685, 3.19344289472,  12146.66705610760 ),
  { 4310  261 }  (         0.675, 0.96179233959,  10454.50138660520 ),
  { 4310  262 }  (         0.648, 1.46327342555,   6268.84875598980 ),
  { 4310  263 }  (         0.589, 2.50543543638,   3097.88382272579 ),
  { 4310  264 }  (         0.551, 5.28099026956,   9388.00590941520 ),
  { 4310  265 }  (         0.696, 3.65342150016,   4804.20927592700 ),
  { 4310  266 }  (         0.669, 2.51030077026,   2388.89402044920 ),
  { 4310  267 }  (         0.550, 0.06883864342,  20199.09495963300 ),
  { 4310  268 }  (         0.629, 4.13350995675,  45892.73043315699 ),
  { 4310  269 }  (         0.678, 6.09190163533,    135.62532501000 ),
  { 4310  270 }  (         0.593, 1.50136257618, 226858.23855437008 ),
  { 4310  271 }  (         0.542, 3.58573645173,   6148.01076995600 ),
  { 4310  272 }  (         0.682, 5.02203067788,  17253.04110768959 ),
  { 4310  273 }  (         0.565, 4.29309238610,  11933.36796066960 ),
  { 4310  274 }  (         0.486, 0.77746204893,     27.40155609680 ),
  { 4310  275 }  (         0.503, 0.58963565969,  15671.08175940660 ),
  { 4310  276 }  (         0.616, 4.06539884128,    227.47613278900 ),
  { 4310  277 }  (         0.583, 6.12695541996,  18875.52586977400 ),
  { 4310  278 }  (         0.537, 2.15056440980,  21954.15760939799 ),
  { 4310  279 }  (         0.669, 6.06986269566,  47162.51635463520 ),
  { 4310  280 }  (         0.475, 0.40343842110,   6915.85958930460 ),
  { 4310  281 }  (         0.540, 2.83444222174,   5326.78669402080 ),
  { 4310  282 }  (         0.530, 5.26359885263,  10988.80815753500 ),
  { 4310  283 }  (         0.582, 3.24533095664,    153.77881048480 ),
  { 4310  284 }  (         0.641, 3.24711791371,   2107.03450754240 ),
  { 4310  285 }  (         0.621, 3.09698523779,  33019.02111220460 ),
  { 4310  286 }  (         0.466, 3.14982372198,  10440.27429260360 ),
  { 4310  287 }  (         0.466, 0.90708835657,   5966.68398033480 ),
  { 4310  288 }  (         0.528, 0.81926454470,    813.55028395980 ),
  { 4310  289 }  (         0.603, 3.81378921927, 316428.22867391503 ),
  { 4310  290 }  (         0.559, 1.81894804124,  17996.03116822220 ),
  { 4310  291 }  (         0.437, 2.28625594435,   6303.85124548380 ),
  { 4310  292 }  (         0.518, 4.86069178322,  20597.24396304120 ),
  { 4310  293 }  (         0.424, 6.23520018693,   6489.26139842860 ),
  { 4310  294 }  (         0.518, 6.17617826756,      0.24381748350 ),
  { 4310  295 }  (         0.404, 5.72804304258,   5642.19824260920 ),
  { 4310  296 }  (         0.458, 1.34117773915,   6287.00800325450 ),
  { 4310  297 }  (         0.548, 5.68454458320, 155427.54293624099 ),
  { 4310  298 }  (         0.547, 1.03391472061,   3646.35037735440 ),
  { 4310  299 }  (         0.428, 4.69800981138,    846.08283475120 ),
  { 4310  300 }  (         0.413, 6.02520699406,   6279.48542133960 ),
  { 4310  301 }  (         0.534, 3.03030638223,  66567.48586525429 ),
  { 4310  302 }  (         0.383, 1.49056949125,  19800.94595622480 ),
  { 4310  303 }  (         0.410, 5.28319622279,  18451.07854656599 ),
  { 4310  304 }  (         0.352, 4.68891600359,   4907.30205014560 ),
  { 4310  305 }  (         0.480, 5.36572651091,    348.92442044800 ),
  { 4310  306 }  (         0.344, 5.89157452896,   6546.15977336420 ),
  { 4310  307 }  (         0.340, 0.37557426440,  13119.72110282519 ),
  { 4310  308 }  (         0.434, 4.98417785901,   6702.56049386660 ),
  { 4310  309 }  (         0.332, 2.68902519126,  29296.61538957860 ),
  { 4310  310 }  (         0.448, 2.16478480251,   5905.70224207560 ),
  { 4310  311 }  (         0.344, 2.06546633735,     49.75702547180 ),
  { 4310  312 }  (         0.315, 1.24023811803,   4061.21921539440 ),
  { 4310  313 }  (         0.324, 2.30897526929,   5017.50837136500 ),
  { 4310  314 }  (         0.413, 0.17171692962,   6286.66627864320 ),
  { 4310  315 }  (         0.431, 3.86601101393,  12489.88562870720 ),
  { 4310  316 }  (         0.349, 4.55372342974,   4933.20844033260 ),
  { 4310  317 }  (         0.323, 0.41971136084,  10770.89325626180 ),
  { 4310  318 }  (         0.341, 2.68612860807,     11.04570026390 ),
  { 4310  319 }  (         0.316, 3.52936906658,  17782.73207278420 ),
  { 4310  320 }  (         0.315, 5.63357264999,    568.82187402740 ),
  { 4310  321 }  (         0.340, 3.83571212349,  10660.68693504240 ),
  { 4310  322 }  (         0.297, 0.62691416712,  20995.39296644940 ),
  { 4310  323 }  (         0.405, 1.00085779471,  16460.33352952499 ),
  { 4310  324 }  (         0.414, 1.21998752076,  51092.72605085480 ),
  { 4310  325 }  (         0.336, 4.71465945226,   6179.98307577280 ),
  { 4310  326 }  (         0.361, 3.71227508354,  28237.23345938940 ),
  { 4310  327 }  (         0.385, 6.21925225757,  24356.78078864160 ),
  { 4310  328 }  (         0.327, 1.05606504715,  11919.14086666800 ),
  { 4310  329 }  (         0.327, 6.14222420989,   6254.62666252360 ),
  { 4310  330 }  (         0.268, 2.47224339737,    664.75604513000 ),
  { 4310  331 }  (         0.269, 1.86207884109,  23141.55838292460 ),
  { 4310  332 }  (         0.345, 0.93461290184,   6058.73105428950 ),
  { 4310  333 }  (         0.296, 4.51687557180,   6418.14093002680 ),
  { 4310  334 }  (         0.353, 4.50033653082,  36949.23080842420 ),
  { 4310  335 }  (         0.260, 4.04963546305,   6525.80445396540 ),
  { 4310  336 }  (         0.298, 2.20046722622, 156137.47598479928 ),
  { 4310  337 }  (         0.253, 3.49900838384,  29864.33402730900 ),
  { 4310  338 }  (         0.254, 2.44901693835,   5331.35744374080 ),
  { 4310  339 }  (         0.296, 0.84347588787,   5729.50644714900 ),
  { 4310  340 }  (         0.298, 1.29194706125,  22805.73556599360 ),
  { 4310  341 }  (         0.241, 2.00721280805,  16737.57723659660 ),
  { 4310  342 }  (         0.311, 1.23668016334,   6281.59137728310 ),
  { 4310  343 }  (         0.240, 2.51650377121,   6245.04817735560 ),
  { 4310  344 }  (         0.332, 3.55576945724,   7668.63742494250 ),
  { 4310  345 }  (         0.264, 4.44052061202,  12964.30070339100 ),
  { 4310  346 }  (         0.257, 1.79654471948,  11080.17157891760 ),
  { 4310  347 }  (         0.260, 3.33077598420,   5888.44996493220 ),
  { 4310  348 }  (         0.285, 0.30886361430,  11823.16163945020 ),
  { 4310  349 }  (         0.290, 5.70141882483,     77.67377042800 ),
  { 4310  350 }  (         0.255, 4.00939664440,   5881.40372823420 ),
  { 4310  351 }  (         0.253, 4.73318493678,  16723.35014259500 ),
  { 4310  352 }  (         0.228, 0.95333661324,   5540.08578945880 ),
  { 4310  353 }  (         0.319, 1.38633229189, 163096.18036118349 ),
  { 4310  354 }  (         0.224, 1.65156322696,  10027.90319572920 ),
  { 4310  355 }  (         0.226, 0.34106460604,  17796.95916678580 ),
  { 4310  356 }  (         0.236, 4.19817431922,     19.66976089979 ),
  { 4310  357 }  (         0.280, 4.14080268970,  12539.85338018300 ),
  { 4310  358 }  (         0.275, 5.50306930248,     32.53255079140 ),
  { 4310  359 }  (         0.223, 5.23334210294,     56.89837493560 ),
  { 4310  360 }  (         0.217, 6.08587881787,   6805.65326808520 ),
  { 4310  361 }  (         0.280, 4.52472044653,   6016.46880826960 ),
  { 4310  362 }  (         0.227, 5.06509843737,   6277.55292568400 ),
  { 4310  363 }  (         0.226, 5.17755154305,  11720.06886523160 ),
  { 4310  364 }  (         0.245, 3.96486270306,     22.77520145080 ),
  { 4310  365 }  (         0.220, 4.72078081970,      6.62855890001 ),
  { 4310  366 }  (         0.207, 5.71701403951,     41.55079098480 ),
  { 4310  367 }  (         0.204, 3.91227411250,   2699.73481931760 ),
  { 4310  368 }  (         0.209, 0.86881969011,   6321.10352262720 ),
  { 4310  369 }  (         0.200, 2.11984445273,   4274.51831083240 ),
  { 4310  370 }  (         0.200, 5.39839888163,   6019.99192661860 ),
  { 4310  371 }  (         0.209, 5.67606291663,  11293.47067435560 ),
  { 4310  372 }  (         0.252, 1.64965729351,   9380.95967271720 ),
  { 4310  373 }  (         0.275, 5.04826903506,     73.29712585900 ),
  { 4310  374 }  (         0.208, 1.88207277133,  11300.58422135640 ),
  { 4310  375 }  (         0.272, 0.74640926842,   1975.49254585600 ),
  { 4310  376 }  (         0.199, 3.30836672397,  22743.40937951640 ),
  { 4310  377 }  (         0.269, 4.48560812155,  64471.99124174489 ),
  { 4310  378 }  (         0.192, 2.17464236325,   5863.59120611620 ),
  { 4310  379 }  (         0.228, 5.85373115869,    128.01884333740 ),
  { 4310  380 }  (         0.261, 2.64321183295,  55022.93574707440 ),
  { 4310  381 }  (         0.220, 5.75012110079,     29.42950853600 ),
  { 4310  382 }  (         0.187, 4.03230554718,    467.96499035440 ),
  { 4310  383 }  (         0.200, 5.60556112058,   1066.49547719000 ),
  { 4310  384 }  (         0.231, 1.09802712785,  12341.80690428090 ),
  { 4310  385 }  (         0.199, 0.29500625200,    149.56319713460 ),
  { 4310  386 }  (         0.249, 5.10473210814,   7875.67186362420 ),
  { 4310  387 }  (         0.208, 0.93013835019,  14919.01785375460 ),
  { 4310  388 }  (         0.179, 0.87104393079,  12721.57209941700 ),
  { 4310  389 }  (         0.203, 1.56920753653,  28286.99048486120 ),
  { 4310  390 }  (         0.179, 2.47036386443,  16062.18452611680 ),
  { 4310  391 }  (         0.198, 3.54061588502,     30.91412563500 ),
  { 4310  392 }  (         0.171, 3.45356518113,   5327.47610838280 ),
  { 4310  393 }  (         0.183, 0.72325421604,   6272.03014972750 ),
  { 4310  394 }  (         0.216, 2.97174580686,  19402.79695281660 ),
  { 4310  395 }  (         0.168, 2.51550550242,  23937.85638974100 ),
  { 4310  396 }  (         0.195, 0.09045393425,    156.40072050240 ),
  { 4310  397 }  (         0.179, 4.49471798090,  31415.37924995700 ),
  { 4310  398 }  (         0.216, 0.42177594328,  23539.70738633280 ),
  { 4310  399 }  (         0.189, 0.37542530191,   9814.60410029120 ),
  { 4310  400 }  (         0.218, 2.36835880025,  16627.37091537720 ),
  { 4310  401 }  (         0.166, 4.23182968446,  16840.67001081519 ),
  { 4310  402 }  (         0.200, 2.02153258098,  16097.67995028260 ),
  { 4310  403 }  (         0.169, 0.91318727000,     95.97922721780 ),
  { 4310  404 }  (         0.211, 5.73370637657,    151.89728108520 ),
  { 4310  405 }  (         0.204, 0.42643085174,    515.46387109300 ),
  { 4310  406 }  (         0.212, 3.00233538977,  12043.57428188900 ),
  { 4310  407 }  (         0.192, 5.46153589821,   6379.05507720920 ),
  { 4310  408 }  (         0.165, 1.38698167064,   4171.42553661380 ),
  { 4310  409 }  (         0.160, 6.23798383332,    202.25339517410 ),
  { 4310  410 }  (         0.215, 0.20889073407,   5621.84292321040 ),
  { 4310  411 }  (         0.181, 4.12439203622,  13341.67431130680 ),
  { 4310  412 }  (         0.153, 1.24460848836,  29826.30635467320 ),
  { 4310  413 }  (         0.150, 3.12999753018,    799.82112516540 ),
  { 4310  414 }  (         0.175, 4.55671604437, 239424.39025435288 ),
  { 4310  415 }  (         0.192, 1.33928820063,    394.62588505920 ),
  { 4310  416 }  (         0.149, 2.65697593276,     21.33564046700 ),
  { 4310  417 }  (         0.146, 5.58021191726,    412.37109687440 ),
  { 4310  418 }  (         0.156, 3.75650175503,  12323.42309600880 ),
  { 4310  419 }  (         0.143, 3.75708566606,  58864.54391814630 ),
  { 4310  420 }  (         0.143, 3.28248547724,     29.82143814880 ),
  { 4310  421 }  (         0.144, 1.07862546598,   1265.56747862640 ),
  { 4310  422 }  (         0.148, 0.23389236655,  10021.83728009940 ),
  { 4310  423 }  (         0.193, 5.92751083086,  40879.44050464380 ),
  { 4310  424 }  (         0.140, 4.97612440269,    158.94351778320 ),
  { 4310  425 }  (         0.148, 2.61640453469,  17157.06188047180 ),
  { 4310  426 }  (         0.141, 3.66871308723,  26084.02180621620 ),
  { 4310  427 }  (         0.147, 5.09968173403,    661.23292678100 ),
  { 4310  428 }  (         0.146, 4.96885605695,  57375.80190084620 ),
  { 4310  429 }  (         0.142, 0.78678347839,  12779.45079542080 ),
  { 4310  430 }  (         0.134, 4.79432636012,    111.18664228760 ),
  { 4310  431 }  (         0.140, 1.27748013377,    107.66352393860 ),
  { 4310  432 }  (         0.169, 2.74893543762,  26735.94526221320 ),
  { 4310  433 }  (         0.165, 3.95288000638,   6357.85744855870 ),
  { 4310  434 }  (         0.183, 5.43418358741,    369.69981594040 ),
  { 4310  435 }  (         0.134, 3.09132862833,     17.81252211800 ),
  { 4310  436 }  (         0.132, 3.05633896779,  22490.96212149340 ),
  { 4310  437 }  (         0.134, 4.09472795832,   6599.46771964800 ),
  { 4310  438 }  (         0.181, 4.22950689891,    966.97087743560 ),
  { 4310  439 }  (         0.152, 5.28885894415,  12669.24447420140 ),
  { 4310  440 }  (         0.150, 5.86819430908,  97238.62754448749 ),
  { 4310  441 }  (         0.142, 5.87266532526,  22476.73502749179 ),
  { 4310  442 }  (         0.145, 5.07330784304,     87.30820453981 ),
  { 4310  443 }  (         0.133, 5.65471067133,     31.97230581680 ),
  { 4310  444 }  (         0.124, 2.83326217072,  12566.21901028560 ),
  { 4310  445 }  (         0.135, 3.12861731644,  32217.20018108080 ),
  { 4310  446 }  (         0.137, 0.86487461904,   9924.81042151060 ),
  { 4310  447 }  (         0.172, 1.98369595114, 174242.46596404970 ),
  { 4310  448 }  (         0.170, 4.41115280254, 327574.51427678125 ),
  { 4310  449 }  (         0.151, 0.46542099527,  39609.65458316560 ),
  { 4310  450 }  (         0.148, 2.13439571118,    491.66329245880 ),
  { 4310  451 }  (         0.153, 3.78801830344,  17363.24742890899 ),
  { 4310  452 }  (         0.165, 5.31654110459,  16943.76278503380 ),
  { 4310  453 }  (         0.165, 4.06747587817,  58953.14544329400 ),
  { 4310  454 }  (         0.118, 0.63846333239,      6.06591562980 ),
  { 4310  455 }  (         0.159, 0.86086959274, 221995.02880149524 ),
  { 4310  456 }  (         0.119, 5.96432932413,   1385.89527633620 ),
  { 4310  457 }  (         0.114, 5.16516114595,  25685.87280280800 ),
  { 4310  458 }  (         0.112, 3.39403722178,  21393.54196985760 ),
  { 4310  459 }  (         0.112, 4.92889233335,     56.80326216980 ),
  { 4310  460 }  (         0.119, 2.40637635942,  18635.92845453620 ),
  { 4310  461 }  (         0.115, 0.23374479051,    418.92439890060 ),
  { 4310  462 }  (         0.122, 0.93575234049,  24492.40611365159 ),
  { 4310  463 }  (         0.115, 4.58880032176,  26709.64694241340 ),
  { 4310  464 }  (         0.130, 4.85539251000,  22345.26037610820 ),
  { 4310  465 }  (         0.140, 1.09413073202,  44809.65020086340 ),
  { 4310  466 }  (         0.112, 6.05401806281,    433.71173787680 ),
  { 4310  467 }  (         0.104, 1.54931540602,    127.95153303460 ),
  { 4310  468 }  (         0.105, 4.82620858888,  33794.54372352860 ),
  { 4310  469 }  (         0.102, 4.12448497391,  15664.03552270859 ),
  { 4310  470 }  (         0.107, 4.67919356465,  77690.75950573849 ),
  { 4310  471 }  (         0.118, 4.52320170120,  19004.64794940840 ),
  { 4310  472 }  (         0.107, 5.71774478555,  77736.78343050249 ),
  { 4310  473 }  (         0.143, 1.81201813018,   4214.06901508480 ),
  { 4310  474 }  (         0.125, 1.14419195615,    625.67019231240 ),
  { 4310  475 }  (         0.124, 3.27736514057,  12566.08438968000 ),
  { 4310  476 }  (         0.110, 1.08682570828,   2787.04302385740 ),
  { 4310  477 }  (         0.105, 1.78318141871,  18139.29450141590 ),
  { 4310  478 }  (         0.102, 4.75119578149,  12242.64628332540 ),
  { 4310  479 }  (         0.137, 1.43510636754,  86464.61331683119 ),
  { 4310  480 }  (         0.101, 4.91289409429,    401.67212175720 ),
  { 4310  481 }  (         0.129, 1.23567904485,  12029.34718788740 ),
  { 4310  482 }  (         0.138, 2.45654707999,   7576.56007357400 ),
  { 4310  483 }  (         0.103, 0.40004073416,  90279.92316810328 ),
  { 4310  484 }  (         0.108, 0.98989774940,   5636.06501667660 ),
  { 4310  485 }  (         0.117, 5.17362872063,  34520.30930938080 ),
  { 4310  486 }  (         0.100, 3.95534628189,   5547.19933645960 ),
  { 4310  487 }  (         0.098, 1.28118280598,  21548.96236929180 ),
  { 4310  488 }  (         0.097, 3.34717130592,  16310.97904572060 ),
  { 4310  489 }  (         0.098, 4.37041908717,  34513.26307268280 ),
  { 4310  490 }  (         0.125, 2.72164432960,  24065.80792277559 ),
  { 4310  491 }  (         0.102, 0.66938025772,  10239.58386601080 ),
  { 4310  492 }  (         0.119, 1.21689479331,   1478.86657406440 ),
  { 4310  493 }  (         0.094, 1.99595224256,  13362.44970679920 ),
  { 4310  494 }  (         0.094, 4.30965982872,  26880.31981303260 ),
  { 4310  495 }  (         0.095, 2.89807657534,  34911.41207609100 ),
  { 4310  496 }  (         0.106, 1.00156653590,  16522.65971600220 ),
  { 4310  497 }  (         0.097, 0.89642320201,  71980.63357473118 ),
  { 4310  498 }  (         0.116, 4.19967201116,    206.70073729660 ),
  { 4310  499 }  (         0.099, 1.37437847718,   1039.02661079040 ),
  { 4310  500 }  (         0.126, 3.21642544972, 305281.94307104882 ),
  { 4310  501 }  (         0.094, 0.68997876060,   7834.12107263940 ),
  { 4310  502 }  (         0.094, 5.58132218606,   3104.93005942380 ),
  { 4310  503 }  (         0.095, 3.03823741110,   8982.81066930900 ),
  { 4310  504 }  (         0.108, 0.52696637156,    276.74577186440 ),
  { 4310  505 }  (         0.124, 3.43899862683, 172146.97134054029 ),
  { 4310  506 }  (         0.102, 1.04031728553,  95143.13292097810 ),
  { 4310  507 }  (         0.104, 3.39218586218,    290.97286586600 ),
  { 4310  508 }  (         0.110, 3.68205877433,  22380.75580027400 ),
  { 4310  509 }  (         0.117, 0.78475956902,  83286.91426955358 ),
  { 4310  510 }  (         0.083, 0.18241793425,  15141.39079431200 ),
  { 4310  511 }  (         0.089, 4.45371820659,    792.77488846740 ),
  { 4310  512 }  (         0.082, 4.80703651241,   6819.88036208680 ),
  { 4310  513 }  (         0.087, 3.43122851097,  27707.54249429480 ),
  { 4310  514 }  (         0.101, 5.32081603011,   2301.58581590939 ),
  { 4310  515 }  (         0.082, 0.87060089842,  10241.20229116720 ),
  { 4310  516 }  (         0.086, 4.61919461931,  36147.40987730040 ),
  { 4310  517 }  (         0.095, 2.87032884659,  23020.65308658799 ),
  { 4310  518 }  (         0.088, 3.21133165690,  33326.57873317420 ),
  { 4310  519 }  (         0.080, 1.84900424847,  21424.46664430340 ),
  { 4310  520 }  (         0.101, 4.18796434479,  30666.15495843280 ),
  { 4310  521 }  (         0.107, 5.77864921649,  34115.11406927460 ),
  { 4310  522 }  (         0.104, 1.08739495962,   6288.59877429880 ),
  { 4310  523 }  (         0.110, 3.32898859416,  72140.62866668739 ),
  { 4310  524 }  (         0.087, 4.40657711727,    142.17862703620 ),
  { 4310  525 }  (         0.109, 1.94546030825,  24279.10701821359 ),
  { 4310  526 }  (         0.087, 4.32472045435,    742.99006053260 ),
  { 4310  527 }  (         0.107, 4.91580912547,    277.03499374140 ),
  { 4310  528 }  (         0.088, 2.10180220766,  26482.17080962440 ),
  { 4310  529 }  (         0.086, 4.01887374432,  12491.37010141550 ),
  { 4310  530 }  (         0.106, 5.49092372854,  62883.35513951360 ),
  { 4310  531 }  (         0.080, 6.19781316983,   6709.67404086740 ),
  { 4310  532 }  (         0.088, 2.09872810657, 238004.52415723629 ),
  { 4310  533 }  (         0.083, 4.90662164029,     51.28033786241 ),
  { 4310  534 }  (         0.095, 4.13387406591,  18216.44381066100 ),
  { 4310  535 }  (         0.078, 6.06949391680, 148434.53403769129 ),
  { 4310  536 }  (         0.079, 3.03048221644,    838.96928775040 ),
  { 4310  537 }  (         0.074, 5.49813051211,  29026.48522950779 ),
  { 4310  538 }  (         0.073, 3.05008665738,    567.71863773040 ),
  { 4310  539 }  (         0.084, 0.46604373274,     45.14121963660 ),
  { 4310  540 }  (         0.093, 2.52267536308,  48739.85989708300 ),
  { 4310  541 }  (         0.076, 1.76418124905,  41654.96311596780 ),
  { 4310  542 }  (         0.067, 5.77851227793,   6311.52503745920 ),
  { 4310  543 }  (         0.062, 3.32967880172,  15508.61512327440 ),
  { 4310  544 }  (         0.079, 5.59773841328,  71960.38658322369 ),
  { 4310  545 }  (         0.057, 3.90629505268,   5999.21653112620 ),
  { 4310  546 }  (         0.061, 0.05695043232,   7856.89627409019 ),
  { 4310  547 }  (         0.061, 5.63297958433,   7863.94251078820 ),
  { 4310  548 }  (         0.065, 3.72178394016,  12573.26524698360 ),
  { 4310  549 }  (         0.057, 4.18217219541,  26087.90314157420 ),
  { 4310  550 }  (         0.066, 3.92262333487,  69853.35207568129 ),
  { 4310  551 }  (         0.053, 5.51119362045,  77710.24834977149 ),
  { 4310  552 }  (         0.053, 4.88573986961,  77717.29458646949 ),
  { 4310  553 }  (         0.062, 2.88876342225,   9411.46461508720 ),
  { 4310  554 }  (         0.051, 1.12657183874,  82576.98122099529 ),
  { 4310  555 }  (         0.045, 2.95671076719,  24602.61243487099 ),
  { 4310  556 }  (         0.040, 5.55145719241,  12565.17137891460 ),
  { 4310  557 }  (         0.039, 1.20838190039,  18842.11400297339 ),
  { 4310  558 }  (         0.045, 3.18590558749,  45585.17281218740 ),
  { 4310  559 }  (         0.049, 2.44790934886,  13613.80427733600 )
  (*$endif *)
      );
  (*$ifdef meeus *)
  vsop87_ear_l1:array[0.. 33,0..2] of extended = (
  (*$else *)
  vsop87_ear_l1:array[0..340,0..2] of extended = (
  (*$endif *)
  { 4311    1 }(628331966747.491, 0.00000000000,      0.00000000000 ),
  { 4311    2 }  (    206058.863, 2.67823455584,   6283.07584999140 ),
  { 4311    3 }  (      4303.430, 2.63512650414,  12566.15169998280 ),
  { 4311    4 }  (       425.264, 1.59046980729,      3.52311834900 ),
  { 4311    5 }  (       108.977, 2.96618001993,   1577.34354244780 ),
  { 4311    6 }  (       093.478, 2.59212835365,  18849.22754997420 ),
  { 4311    7 }  (       119.261, 5.79557487799,     26.29831979980 ),
  { 4311    8 }  (        72.122, 1.13846158196,    529.69096509460 ),
  { 4311    9 }  (        67.768, 1.87472304791,    398.14900340820 ),
  { 4311   10 }  (        67.327, 4.40918235168,   5507.55323866740 ),
  { 4311   11 }  (        59.027, 2.88797038460,   5223.69391980220 ),
  { 4311   12 }  (        55.976, 2.17471680261,    155.42039943420 ),
  { 4311   13 }  (        45.407, 0.39803079805,    796.29800681640 ),
  { 4311   14 }  (        36.369, 0.46624739835,    775.52261132400 ),
  { 4311   15 }  (        28.958, 2.64707383882,      7.11354700080 ),
  { 4311   16 }  (        19.097, 1.84628332577,   5486.77784317500 ),
  { 4311   17 }  (        20.844, 5.34138275149,      0.98032106820 ),
  { 4311   18 }  (        18.508, 4.96855124577,    213.29909543800 ),
  { 4311   19 }  (        16.233, 0.03216483047,   2544.31441988340 ),
  { 4311   20 }  (        17.293, 2.99116864949,   6275.96230299060 ),
  { 4311   21 }  (        15.832, 1.43049285325,   2146.16541647520 ),
  { 4311   22 }  (        14.615, 1.20532366323,  10977.07880469900 ),
  { 4311   23 }  (        11.877, 3.25804815607,   5088.62883976680 ),
  { 4311   24 }  (        11.514, 2.07502418155,   4694.00295470760 ),
  { 4311   25 }  (         9.721, 4.23925472239,   1349.86740965880 ),
  { 4311   26 }  (         9.969, 1.30262991097,   6286.59896834040 ),
  { 4311   27 }  (         9.452, 2.69957062864,    242.72860397400 ),
  { 4311   28 }  (        12.461, 2.83432285512,   1748.01641306700 ),
  { 4311   29 }  (        11.808, 5.27379790480,   1194.44701022460 ),
  { 4311   30 }  (         8.577, 5.64475868067,    951.71840625060 ),
  { 4311   31 }  (        10.641, 0.76614199202,    553.56940284240 ),
  { 4311   32 }  (         7.576, 5.30062664886,   2352.86615377180 ),
  (*$ifndef meeus *)
  { 4311   33 }  (         5.834, 1.76649917904,   1059.38193018920 ),
  (*$endif *)
  { 4311   34 }  (         6.385, 2.65033984967,   9437.76293488700 ),
  (*$ifndef meeus *)
  { 4311   35 }  (         5.223, 5.66135767624,  71430.69561812909 ),
  { 4311   36 }  (         5.305, 0.90857521574,   3154.68708489560 ),
  (*$endif *)
  { 4311   37 }  (         6.101, 4.66632584188,   4690.47983635860 )
  (*$ifndef meeus *)                                                 ,
  { 4311   38 }  (         4.330, 0.24102555403,   6812.76681508600 ),
  { 4311   39 }  (         5.041, 1.42490103709,   6438.49624942560 ),
  { 4311   40 }  (         4.259, 0.77355900599,  10447.38783960440 ),
  { 4311   41 }  (         5.198, 1.85353197345,    801.82093112380 ),
  { 4311   42 }  (         3.744, 2.00119516488,   8031.09226305840 ),
  { 4311   43 }  (         3.558, 2.42901552681,  14143.49524243060 ),
  { 4311   44 }  (         3.372, 3.86210700128,   1592.59601363280 ),
  { 4311   45 }  (         3.374, 0.88776219727,  12036.46073488820 ),
  { 4311   46 }  (         3.175, 3.18785710594,   4705.73230754360 ),
  { 4311   47 }  (         3.221, 0.61599835472,   8429.24126646660 ),
  { 4311   48 }  (         4.132, 5.23992859705,   7084.89678111520 ),
  { 4311   49 }  (         2.970, 6.07026318493,   4292.33083295040 ),
  { 4311   50 }  (         2.900, 2.32464208411,     20.35531939880 ),
  { 4311   51 }  (         3.504, 4.79975694359,   6279.55273164240 ),
  { 4311   52 }  (         2.950, 1.43108874817,   5746.27133789600 ),
  { 4311   53 }  (         2.697, 4.80368225199,   7234.79425624200 ),
  { 4311   54 }  (         2.531, 6.22290682655,   6836.64525283380 ),
  { 4311   55 }  (         2.745, 0.93466065396,   5760.49843189760 ),
  { 4311   56 }  (         3.250, 3.39954640038,   7632.94325965020 ),
  { 4311   57 }  (         2.277, 5.00277837672,  17789.84561978500 ),
  { 4311   58 }  (         2.075, 3.95534978634,  10213.28554621100 ),
  { 4311   59 }  (         2.061, 2.22411683077,   5856.47765911540 ),
  { 4311   60 }  (         2.252, 5.67166499885,  11499.65622279280 ),
  { 4311   61 }  (         2.148, 5.20184578235,  11513.88331679440 ),
  { 4311   62 }  (         1.886, 0.53198320577,   3340.61242669980 ),
  { 4311   63 }  (         1.875, 4.73511970207,  83996.84731811189 ),
  { 4311   64 }  (         2.060, 2.54987293999,  25132.30339996560 ),
  { 4311   65 }  (         1.794, 1.47435409831,   4164.31198961300 ),
  { 4311   66 }  (         1.778, 3.02473091781,      5.52292430740 ),
  { 4311   67 }  (         2.029, 0.90960209983,   6256.77753019160 ),
  { 4311   68 }  (         2.075, 2.26767270157,    522.57741809380 ),
  { 4311   69 }  (         1.772, 3.02622802353,   5753.38488489680 ),
  { 4311   70 }  (         1.569, 6.12410242782,   5216.58037280140 ),
  { 4311   71 }  (         1.590, 4.63713748247,      3.28635741780 ),
  { 4311   72 }  (         1.542, 4.20004448567,  13367.97263110660 ),
  { 4311   73 }  (         1.427, 1.19088061711,   3894.18182954220 ),
  { 4311   74 }  (         1.375, 3.09301252193,    135.06508003540 ),
  { 4311   75 }  (         1.359, 4.24532506641,    426.59819087600 ),
  { 4311   76 }  (         1.340, 5.76511818622,   6040.34724601740 ),
  { 4311   77 }  (         1.284, 3.08524663344,   5643.17856367740 ),
  { 4311   78 }  (         1.250, 3.07748157144,  11926.25441366880 ),
  { 4311   79 }  (         1.551, 3.07665451458,   6681.22485339960 ),
  { 4311   80 }  (         1.268, 2.09196018331,   6290.18939699220 ),
  { 4311   81 }  (         1.144, 3.24444699514,  12168.00269657460 ),
  { 4311   82 }  (         1.248, 3.44504937285,    536.80451209540 ),
  { 4311   83 }  (         1.118, 2.31829670425,  16730.46368959580 ),
  { 4311   84 }  (         1.105, 5.31966001019,     23.87843774780 ),
  { 4311   85 }  (         1.051, 3.75015946014,   7860.41939243920 ),
  { 4311   86 }  (         1.025, 2.44688534235,   1990.74501704100 ),
  { 4311   87 }  (         0.962, 0.81771017882,      3.88133535800 ),
  { 4311   88 }  (         0.910, 0.41727865299,   7079.37385680780 ),
  { 4311   89 }  (         0.883, 5.16833917651,  11790.62908865880 ),
  { 4311   90 }  (         0.957, 4.07673573735,   6127.65545055720 ),
  { 4311   91 }  (         1.110, 3.90096793825,  11506.76976979360 ),
  { 4311   92 }  (         0.802, 3.88778875582,  10973.55568635000 ),
  { 4311   93 }  (         0.780, 2.39934293755,   1589.07289528380 ),
  { 4311   94 }  (         0.758, 1.30034364248,    103.09277421860 ),
  { 4311   95 }  (         0.749, 4.96275803300,   6496.37494542940 ),
  { 4311   96 }  (         0.765, 3.36312388424,     36.02786667740 ),
  { 4311   97 }  (         0.915, 5.41543742089,    206.18554843720 ),
  { 4311   98 }  (         0.776, 2.57589093871,  11371.70468975820 ),
  { 4311   99 }  (         0.772, 3.98369209464,    955.59974160860 ),
  { 4311  100 }  (         0.749, 5.17890001805,  10969.96525769820 ),
  { 4311  101 }  (         0.806, 0.34218864254,   9917.69687450980 ),
  { 4311  102 }  (         0.728, 5.20962563787,     38.02767263580 ),
  { 4311  103 }  (         0.685, 2.77592961854,     20.77539549240 ),
  { 4311  104 }  (         0.636, 4.28242193632,     28.44918746780 ),
  { 4311  105 }  (         0.608, 5.63278508906,  10984.19235169980 ),
  { 4311  106 }  (         0.704, 5.60738823665,   3738.76143010800 ),
  { 4311  107 }  (         0.685, 0.38876148682,     15.25247118500 ),
  { 4311  108 }  (         0.601, 0.73489602442,    419.48464387520 ),
  { 4311  109 }  (         0.716, 2.65279791438,   6309.37416979120 ),
  { 4311  110 }  (         0.584, 5.54502568227,  17298.18232732620 ),
  { 4311  111 }  (         0.650, 1.13379656406,   7058.59846131540 ),
  { 4311  112 }  (         0.688, 2.59683891779,   3496.03282613400 ),
  { 4311  113 }  (         0.485, 0.44467180946,  12352.85260454480 ),
  { 4311  114 }  (         0.528, 2.74936967681,   3930.20969621960 ),
  { 4311  115 }  (         0.597, 5.27668281777,  10575.40668294180 ),
  { 4311  116 }  (         0.583, 3.18929067810,   4732.03062734340 ),
  { 4311  117 }  (         0.526, 5.01697321546,   5884.92684658320 ),
  { 4311  118 }  (         0.540, 1.29175137075,    640.87760738220 ),
  { 4311  119 }  (         0.473, 5.49953306970,   5230.80746680300 ),
  { 4311  120 }  (         0.406, 5.21248452189,    220.41264243880 ),
  { 4311  121 }  (         0.395, 1.87474483222,  16200.77272450120 ),
  { 4311  122 }  (         0.370, 3.84921354713,  18073.70493865020 ),
  { 4311  123 }  (         0.367, 0.88533542778,   6283.14316029419 ),
  { 4311  124 }  (         0.379, 0.37983009325,  10177.25767953360 ),
  { 4311  125 }  (         0.356, 3.84145204913,  11712.95531823080 ),
  { 4311  126 }  (         0.374, 5.01577520608,      7.04623669800 ),
  { 4311  127 }  (         0.381, 4.30250406634,   6062.66320755260 ),
  { 4311  128 }  (         0.471, 0.86381834647,   6069.77675455340 ),
  { 4311  129 }  (         0.367, 1.32943839763,   6283.00853968860 ),
  { 4311  130 }  (         0.460, 5.19667219575,   6284.05617105960 ),
  { 4311  131 }  (         0.333, 5.54256205741,   4686.88940770680 ),
  { 4311  132 }  (         0.341, 4.36522989934,   7238.67559160000 ),
  { 4311  133 }  (         0.336, 4.00205876835,   3097.88382272579 ),
  { 4311  134 }  (         0.359, 6.22679790284,    245.83164622940 ),
  { 4311  135 }  (         0.307, 2.35299010924,    170.67287061920 ),
  { 4311  136 }  (         0.343, 3.77164927143,   6076.89030155420 ),
  { 4311  137 }  (         0.296, 5.44152227481,  17260.15465469040 ),
  { 4311  138 }  (         0.328, 0.13837875384,  11015.10647733480 ),
  { 4311  139 }  (         0.268, 1.13904550630,  12569.67481833180 ),
  { 4311  140 }  (         0.263, 0.00538633678,   4136.91043351620 ),
  { 4311  141 }  (         0.282, 5.04399837480,   7477.52286021600 ),
  { 4311  142 }  (         0.288, 3.13401177517,  12559.03815298200 ),
  { 4311  143 }  (         0.259, 0.93882269387,   5642.19824260920 ),
  { 4311  144 }  (         0.292, 1.98420020514,  12132.43996210600 ),
  { 4311  145 }  (         0.247, 3.84244798532,   5429.87946823940 ),
  { 4311  146 }  (         0.245, 5.70467521726,  65147.61976813770 ),
  { 4311  147 }  (         0.241, 0.99480969552,   3634.62102451840 ),
  { 4311  148 }  (         0.246, 3.06168069935,    110.20632121940 ),
  { 4311  149 }  (         0.239, 6.11855909114,  11856.21865142450 ),
  { 4311  150 }  (         0.263, 0.66348415419,  21228.39202354580 ),
  { 4311  151 }  (         0.262, 1.51070507866,  12146.66705610760 ),
  { 4311  152 }  (         0.230, 1.75927314884,   9779.10867612540 ),
  { 4311  153 }  (         0.223, 2.00967043606,   6172.86952877200 ),
  { 4311  154 }  (         0.246, 1.10411690865,   6282.09552892320 ),
  { 4311  155 }  (         0.221, 3.03945240854,   8635.94200376320 ),
  { 4311  156 }  (         0.214, 4.03840869663,  14314.16811304980 ),
  { 4311  157 }  (         0.236, 5.46915070580,  13916.01910964160 ),
  { 4311  158 }  (         0.224, 4.68408089456,  24072.92146977640 ),
  { 4311  159 }  (         0.212, 2.13695625494,   5849.36411211460 ),
  { 4311  160 }  (         0.207, 3.07724246401,     11.72935283600 ),
  { 4311  161 }  (         0.207, 6.10306282747,  23543.23050468179 ),
  { 4311  162 }  (         0.266, 1.00709566823,   2388.89402044920 ),
  { 4311  163 }  (         0.217, 6.27837036335,  17267.26820169119 ),
  { 4311  164 }  (         0.204, 2.34615348695,    266.60704172180 ),
  { 4311  165 }  (         0.195, 5.55015549753,   6133.51265285680 ),
  { 4311  166 }  (         0.188, 2.52667166175,   6525.80445396540 ),
  { 4311  167 }  (         0.185, 0.90960768344,  18319.53658487960 ),
  { 4311  168 }  (         0.177, 1.73429218289, 154717.60988768269 ),
  { 4311  169 }  (         0.187, 4.76483647432,   4535.05943692440 ),
  { 4311  170 }  (         0.186, 4.63080493407,  10440.27429260360 ),
  { 4311  171 }  (         0.215, 2.81255454560,   7342.45778018060 ),
  { 4311  172 }  (         0.172, 1.45551888559,   9225.53927328300 ),
  { 4311  173 }  (         0.162, 3.30661909388,    639.89728631400 ),
  { 4311  174 }  (         0.168, 2.17671416605,     27.40155609680 ),
  { 4311  175 }  (         0.160, 1.68164180475,  15110.46611986620 ),
  { 4311  176 }  (         0.158, 0.13519771874,  13095.84266507740 ),
  { 4311  177 }  (         0.183, 0.56281322071,  13517.87010623340 ),
  { 4311  178 }  (         0.179, 3.58450811616,     87.30820453981 ),
  { 4311  179 }  (         0.152, 2.84070476818,   5650.29211067820 ),
  { 4311  180 }  (         0.182, 0.44065530624,  17253.04110768959 ),
  { 4311  181 }  (         0.160, 5.95767264171,   4701.11650170840 ),
  { 4311  182 }  (         0.142, 1.46290137520,  11087.28512591840 ),
  { 4311  183 }  (         0.142, 2.04464036087,  20426.57109242200 ),
  { 4311  184 }  (         0.131, 5.40912137746,   2699.73481931760 ),
  { 4311  185 }  (         0.144, 2.07312090485,  25158.60171976540 ),
  { 4311  186 }  (         0.147, 6.15106982168,   9623.68827669120 ),
  { 4311  187 }  (         0.141, 5.55739979498,  10454.50138660520 ),
  { 4311  188 }  (         0.135, 0.06098110407,  16723.35014259500 ),
  { 4311  189 }  (         0.124, 5.81218025669,  17256.63153634140 ),
  { 4311  190 }  (         0.124, 2.36293551623,   4933.20844033260 ),
  { 4311  191 }  (         0.126, 3.47435905118,  22483.84857449259 ),
  { 4311  192 }  (         0.159, 5.63954754618,   5729.50644714900 ),
  { 4311  193 }  (         0.123, 3.92815963256,  17996.03116822220 ),
  { 4311  194 }  (         0.148, 3.02509280598,   1551.04522264800 ),
  { 4311  195 }  (         0.120, 5.91904349732,   6206.80977871580 ),
  { 4311  196 }  (         0.134, 3.11122937825,  21954.15760939799 ),
  { 4311  197 }  (         0.119, 5.52141123450,    709.93304855830 ),
  { 4311  198 }  (         0.122, 3.00813429479,  19800.94595622480 ),
  { 4311  199 }  (         0.127, 1.37618620001,  14945.31617355440 ),
  { 4311  200 }  (         0.141, 2.56889468729,   1052.26838318840 ),
  { 4311  201 }  (         0.123, 2.83671175442,  11919.14086666800 ),
  { 4311  202 }  (         0.118, 0.81934438215,   5331.35744374080 ),
  { 4311  203 }  (         0.151, 2.68731829165,  11769.85369316640 ),
  { 4311  204 }  (         0.119, 5.08835797638,   5481.25491886760 ),
  { 4311  205 }  (         0.153, 2.46021790779,  11933.36796066960 ),
  { 4311  206 }  (         0.108, 1.04936452145,  11403.67699557500 ),
  { 4311  207 }  (         0.128, 0.99794735107,   8827.39026987480 ),
  { 4311  208 }  (         0.144, 2.54869747042,    227.47613278900 ),
  { 4311  209 }  (         0.150, 4.50631437136,   2379.16447357160 ),
  { 4311  210 }  (         0.107, 1.79272017026,  13119.72110282519 ),
  { 4311  211 }  (         0.107, 4.43556814486,  18422.62935909819 ),
  { 4311  212 }  (         0.109, 0.29269062317,  16737.57723659660 ),
  { 4311  213 }  (         0.141, 3.18979826258,   6262.30045449900 ),
  { 4311  214 }  (         0.122, 4.23040027813,     29.42950853600 ),
  { 4311  215 }  (         0.111, 5.16954029551,  17782.73207278420 ),
  { 4311  216 }  (         0.100, 3.52213872761,  18052.92954315780 ),
  { 4311  217 }  (         0.108, 1.08514212991,  16858.48253293320 ),
  { 4311  218 }  (         0.106, 1.96085248410,     74.78159856730 ),
  { 4311  219 }  (         0.110, 2.30582372873,  16460.33352952499 ),
  { 4311  220 }  (         0.097, 3.50918940210,   5333.90024102160 ),
  { 4311  221 }  (         0.099, 3.56417337974,    735.87651353180 ),
  { 4311  222 }  (         0.094, 5.01857894228,   3128.38876509580 ),
  { 4311  223 }  (         0.097, 1.65579893894,    533.21408344360 ),
  { 4311  224 }  (         0.092, 0.89217162285,  29296.61538957860 ),
  { 4311  225 }  (         0.123, 3.16062050433,   9380.95967271720 ),
  { 4311  226 }  (         0.102, 1.20493500565,  23020.65308658799 ),
  { 4311  227 }  (         0.088, 2.21296088224,  12721.57209941700 ),
  { 4311  228 }  (         0.089, 1.54264720310,  20199.09495963300 ),
  { 4311  229 }  (         0.113, 4.83320707870,  16496.36139620240 ),
  { 4311  230 }  (         0.121, 6.19860353182,   9388.00590941520 ),
  { 4311  231 }  (         0.089, 4.08082274765,  22805.73556599360 ),
  { 4311  232 }  (         0.098, 1.09181832830,  12043.57428188900 ),
  { 4311  233 }  (         0.086, 1.13655027605, 143571.32428481648 ),
  { 4311  234 }  (         0.088, 5.96980472191,    107.66352393860 ),
  { 4311  235 }  (         0.082, 5.01340404594,  22003.91463486980 ),
  { 4311  236 }  (         0.094, 1.69615700473,  23006.42599258639 ),
  { 4311  237 }  (         0.081, 3.00657814365,   2118.76386037840 ),
  { 4311  238 }  (         0.098, 1.39215287161,   8662.24032356300 ),
  { 4311  239 }  (         0.077, 3.33555190840,  15720.83878487840 ),
  { 4311  240 }  (         0.082, 5.86880116464,   2787.04302385740 ),
  { 4311  241 }  (         0.076, 5.67183650604,     14.22709400160 ),
  { 4311  242 }  (         0.081, 6.16619455699,   1039.02661079040 ),
  { 4311  243 }  (         0.076, 3.21449884756,    111.18664228760 ),
  { 4311  244 }  (         0.078, 1.37531518377,  21947.11137270000 ),
  { 4311  245 }  (         0.074, 3.58814195051,  11609.86254401220 ),
  { 4311  246 }  (         0.077, 4.84846488388,  22743.40937951640 ),
  { 4311  247 }  (         0.090, 1.48869013606,  15671.08175940660 ),
  { 4311  248 }  (         0.082, 3.48618399109,  29088.81141598500 ),
  { 4311  249 }  (         0.069, 3.55746476593,   4590.91018048900 ),
  { 4311  250 }  (         0.069, 1.93625656075,    135.62532501000 ),
  { 4311  251 }  (         0.070, 2.66548322237,  18875.52586977400 ),
  { 4311  252 }  (         0.069, 5.41478093731,  26735.94526221320 ),
  { 4311  253 }  (         0.079, 5.15154513662,  12323.42309600880 ),
  { 4311  254 }  (         0.094, 3.62899392448,  77713.77146812050 ),
  { 4311  255 }  (         0.078, 4.17011182047,   1066.49547719000 ),
  { 4311  256 }  (         0.071, 3.89435637865,  22779.43724619380 ),
  { 4311  257 }  (         0.063, 4.53968787714,   8982.81066930900 ),
  { 4311  258 }  (         0.069, 0.96028230548,  14919.01785375460 ),
  { 4311  259 }  (         0.076, 3.29092216589,   2942.46342329160 ),
  { 4311  260 }  (         0.063, 4.09167842893,  16062.18452611680 ),
  { 4311  261 }  (         0.065, 3.34580407184,     51.28033786241 ),
  { 4311  262 }  (         0.065, 5.75757544877,  52670.06959330260 ),
  { 4311  263 }  (         0.068, 5.75884067555,  21424.46664430340 ),
  { 4311  264 }  (         0.057, 5.45122399850,  12592.45001978260 ),
  { 4311  265 }  (         0.057, 5.25043362558,  20995.39296644940 ),
  { 4311  266 }  (         0.073, 0.53299090807,   2301.58581590939 ),
  { 4311  267 }  (         0.070, 4.31243357502,  19402.79695281660 ),
  { 4311  268 }  (         0.067, 2.53852336668,    377.37360791580 ),
  { 4311  269 }  (         0.056, 3.20816844695,  24889.57479599160 ),
  { 4311  270 }  (         0.053, 3.17816599142,  18451.07854656599 ),
  { 4311  271 }  (         0.053, 3.61529270216,     77.67377042800 ),
  { 4311  272 }  (         0.053, 0.45467549335,  30666.15495843280 ),
  { 4311  273 }  (         0.061, 0.14807288453,  23013.53953958720 ),
  { 4311  274 }  (         0.051, 3.32803972907,     56.89837493560 ),
  { 4311  275 }  (         0.052, 3.41177624177,  23141.55838292460 ),
  { 4311  276 }  (         0.058, 3.13638677202,    309.27832265580 ),
  { 4311  277 }  (         0.070, 2.50592323465,  31415.37924995700 ),
  { 4311  278 }  (         0.052, 5.10673376738,  17796.95916678580 ),
  { 4311  279 }  (         0.067, 6.27917920454,  22345.26037610820 ),
  { 4311  280 }  (         0.050, 0.42577644151,  25685.87280280800 ),
  { 4311  281 }  (         0.048, 0.70204553333,   1162.47470440780 ),
  { 4311  282 }  (         0.066, 3.64350022359,  15265.88651930040 ),
  { 4311  283 }  (         0.050, 5.74382917440,     19.66976089979 ),
  { 4311  284 }  (         0.050, 4.69825387775,  28237.23345938940 ),
  { 4311  285 }  (         0.047, 5.74015846442,  12139.55350910680 ),
  { 4311  286 }  (         0.054, 1.97301333704,  23581.25817731760 ),
  { 4311  287 }  (         0.049, 4.98223579027,  10021.83728009940 ),
  { 4311  288 }  (         0.046, 5.41431705539,  33019.02111220460 ),
  { 4311  289 }  (         0.051, 1.23882053879,  12539.85338018300 ),
  { 4311  290 }  (         0.046, 2.41369976086,  98068.53671630539 ),
  { 4311  291 }  (         0.044, 0.80750593746, 167283.76158766549 ),
  { 4311  292 }  (         0.045, 4.39613584445,    433.71173787680 ),
  { 4311  293 }  (         0.044, 2.57358208785,  12964.30070339100 ),
  { 4311  294 }  (         0.046, 0.26142733448,     11.04570026390 ),
  { 4311  295 }  (         0.045, 2.46230645202,  51868.24866217880 ),
  { 4311  296 }  (         0.048, 0.89551707131,  56600.27928952220 ),
  { 4311  297 }  (         0.057, 1.86416707010,  25287.72379939980 ),
  { 4311  298 }  (         0.042, 5.26377513431,  26084.02180621620 ),
  { 4311  299 }  (         0.049, 3.17757670611,   6303.85124548380 ),
  { 4311  300 }  (         0.052, 3.65266055509,   7872.14874527520 ),
  { 4311  301 }  (         0.040, 1.81891629936,  34596.36465465240 ),
  { 4311  302 }  (         0.043, 1.94164978061,   1903.43681250120 ),
  { 4311  303 }  (         0.041, 0.74461854136,  23937.85638974100 ),
  { 4311  304 }  (         0.048, 6.26034008181,  28286.99048486120 ),
  { 4311  305 }  (         0.045, 5.45575017530,  60530.48898574180 ),
  { 4311  306 }  (         0.040, 2.92105728682,  21548.96236929180 ),
  { 4311  307 }  (         0.040, 0.04502010161,  38526.57435087200 ),
  { 4311  308 }  (         0.053, 3.64791042082,  11925.27409260060 ),
  { 4311  309 }  (         0.041, 5.04048954693,  27832.03821928320 ),
  { 4311  310 }  (         0.042, 5.19292937193,  19004.64794940840 ),
  { 4311  311 }  (         0.040, 2.57120233428,  24356.78078864160 ),
  { 4311  312 }  (         0.038, 3.49190341464, 226858.23855437008 ),
  { 4311  313 }  (         0.039, 4.61184303844,     95.97922721780 ),
  { 4311  314 }  (         0.043, 2.20648228147,  13521.75144159140 ),
  { 4311  315 }  (         0.040, 5.83461945819,  16193.65917750039 ),
  { 4311  316 }  (         0.045, 3.73714372195,   7875.67186362420 ),
  { 4311  317 }  (         0.043, 1.14078465002,     49.75702547180 ),
  { 4311  318 }  (         0.037, 1.29390383811,    310.84079886840 ),
  { 4311  319 }  (         0.038, 0.95970925950,    664.75604513000 ),
  { 4311  320 }  (         0.037, 4.27532649462,   6709.67404086740 ),
  { 4311  321 }  (         0.038, 2.20108541046,  28628.33622609960 ),
  { 4311  322 }  (         0.039, 0.85957361635,  16522.65971600220 ),
  { 4311  323 }  (         0.040, 4.35214003837,  48739.85989708300 ),
  { 4311  324 }  (         0.036, 1.68167662194,  10344.29506538580 ),
  { 4311  325 }  (         0.040, 5.13217319067,  15664.03552270859 ),
  { 4311  326 }  (         0.036, 3.72187132496,  30774.50164257480 ),
  { 4311  327 }  (         0.036, 3.32158458257,  16207.88627150200 ),
  { 4311  328 }  (         0.045, 3.94202418608,  10988.80815753500 ),
  { 4311  329 }  (         0.039, 1.51948786199,  12029.34718788740 ),
  { 4311  330 }  (         0.026, 3.87685883180,   6262.72053059260 ),
  { 4311  331 }  (         0.024, 4.91804163466,  19651.04848109800 ),
  { 4311  332 }  (         0.023, 0.29300197709,  13362.44970679920 ),
  { 4311  333 }  (         0.021, 3.18605672363,   6277.55292568400 ),
  { 4311  334 }  (         0.021, 6.07546891132,  18139.29450141590 ),
  { 4311  335 }  (         0.022, 2.31199937177,   6303.43116939020 ),
  { 4311  336 }  (         0.021, 3.58418394393,  18209.33026366019 ),
  { 4311  337 }  (         0.026, 2.06801296900,  12573.26524698360 ),
  { 4311  338 }  (         0.021, 1.56857722317,  13341.67431130680 ),
  { 4311  339 }  (         0.024, 5.72605158675,  29864.33402730900 ),
  { 4311  340 }  (         0.024, 1.40237993205,  14712.31711645800 ),
  { 4311  341 }  (         0.025, 5.71466092822,  25934.12433108940 )
  (*$endif *)
      );
  (*$ifdef meeus *)
  vsop87_ear_l2:array[0.. 19,0..2] of extended = (
  (*$else *)
  vsop87_ear_l2:array[0..141,0..2] of extended = (
  (*$endif *)
  { 4312    1 }  (     52918.870, 0.00000000000,      0.00000000000 ),
  { 4312    2 }  (      8719.837, 1.07209665242,   6283.07584999140 ),
  { 4312    3 }  (       309.125, 0.86728818832,  12566.15169998280 ),
  { 4312    4 }  (        27.339, 0.05297871691,      3.52311834900 ),
  { 4312    5 }  (        16.334, 5.18826691036,     26.29831979980 ),
  { 4312    6 }  (        15.752, 3.68457889430,    155.42039943420 ),
  { 4312    7 }  (         9.541, 0.75742297675,  18849.22754997420 ),
  { 4312    8 }  (         8.937, 2.05705419118,  77713.77146812050 ),
  { 4312    9 }  (         6.952, 0.82673305410,    775.52261132400 ),
  { 4312   10 }  (         5.064, 4.66284525271,   1577.34354244780 ),
  { 4312   11 }  (         4.061, 1.03057162962,      7.11354700080 ),
  { 4312   12 }  (         3.463, 5.14074632811,    796.29800681640 ),
  { 4312   13 }  (         3.169, 6.05291851171,   5507.55323866740 ),
  { 4312   14 }  (         3.020, 1.19246506441,    242.72860397400 ),
  { 4312   15 }  (         2.886, 6.11652627155,    529.69096509460 ),
  { 4312   16 }  (         3.810, 3.44050803490,   5573.14280143310 ),
  { 4312   17 }  (         2.714, 0.30637881025,    398.14900340820 ),
  { 4312   18 }  (         2.371, 4.38118838167,   5223.69391980220 ),
  { 4312   19 }  (         2.538, 2.27992810679,    553.56940284240 ),
  { 4312   20 }  (         2.079, 3.75435330484,      0.98032106820 )
  (*$ifndef meeus *)                                                 ,
  { 4312   21 }  (         1.675, 0.90216407959,    951.71840625060 ),
  { 4312   22 }  (         1.534, 5.75900462759,   1349.86740965880 ),
  { 4312   23 }  (         1.224, 2.97328088405,   2146.16541647520 ),
  { 4312   24 }  (         1.449, 4.36415913970,   1748.01641306700 ),
  { 4312   25 }  (         1.341, 3.72061130861,   1194.44701022460 ),
  { 4312   26 }  (         1.254, 2.94846826628,   6438.49624942560 ),
  { 4312   27 }  (         0.999, 5.98640014468,   6286.59896834040 ),
  { 4312   28 }  (         0.917, 4.79788687522,   5088.62883976680 ),
  { 4312   29 }  (         0.828, 3.31321076572,    213.29909543800 ),
  { 4312   30 }  (         1.103, 1.27104454479, 161000.68573767410 ),
  { 4312   31 }  (         0.762, 3.41582762988,   5486.77784317500 ),
  { 4312   32 }  (         1.044, 0.60409577691,   3154.68708489560 ),
  { 4312   33 }  (         0.887, 5.23465144638,   7084.89678111520 ),
  { 4312   34 }  (         0.645, 1.60096192515,   2544.31441988340 ),
  { 4312   35 }  (         0.681, 3.43155669169,   4694.00295470760 ),
  { 4312   36 }  (         0.605, 2.47806340546,  10977.07880469900 ),
  { 4312   37 }  (         0.706, 6.19393222575,   4690.47983635860 ),
  { 4312   38 }  (         0.643, 1.98042503148,    801.82093112380 ),
  { 4312   39 }  (         0.502, 1.44394375363,   6836.64525283380 ),
  { 4312   40 }  (         0.490, 2.34129524194,   1592.59601363280 ),
  { 4312   41 }  (         0.458, 1.30876448575,   4292.33083295040 ),
  { 4312   42 }  (         0.431, 0.03526421494,   7234.79425624200 ),
  { 4312   43 }  (         0.379, 3.17030522615,   6309.37416979120 ),
  { 4312   44 }  (         0.348, 0.99049550009,   6040.34724601740 ),
  { 4312   45 }  (         0.386, 1.57019797263,  71430.69561812909 ),
  { 4312   46 }  (         0.347, 0.67013291338,   1059.38193018920 ),
  { 4312   47 }  (         0.458, 3.81499443681, 149854.40013480789 ),
  { 4312   48 }  (         0.302, 1.91760044838,  10447.38783960440 ),
  { 4312   49 }  (         0.307, 3.55343347416,   8031.09226305840 ),
  { 4312   50 }  (         0.395, 4.93701776616,   7632.94325965020 ),
  { 4312   51 }  (         0.314, 3.18093696547,   2352.86615377180 ),
  { 4312   52 }  (         0.282, 4.41936437052,   9437.76293488700 ),
  { 4312   53 }  (         0.276, 2.71314254553,   3894.18182954220 ),
  { 4312   54 }  (         0.298, 2.52037474210,   6127.65545055720 ),
  { 4312   55 }  (         0.230, 1.37790215549,   4705.73230754360 ),
  { 4312   56 }  (         0.252, 0.55330133471,   6279.55273164240 ),
  { 4312   57 }  (         0.255, 5.26570187369,   6812.76681508600 ),
  { 4312   58 }  (         0.275, 0.67264264272,  25132.30339996560 ),
  { 4312   59 }  (         0.178, 0.92820785174,   1990.74501704100 ),
  { 4312   60 }  (         0.221, 0.63897368842,   6256.77753019160 ),
  { 4312   61 }  (         0.155, 0.77319790838,  14143.49524243060 ),
  { 4312   62 }  (         0.150, 2.40470465561,    426.59819087600 ),
  { 4312   63 }  (         0.196, 6.06877865012,    640.87760738220 ),
  { 4312   64 }  (         0.137, 2.21679460145,   8429.24126646660 ),
  { 4312   65 }  (         0.127, 3.26094223174,  17789.84561978500 ),
  { 4312   66 }  (         0.128, 5.47237279946,  12036.46073488820 ),
  { 4312   67 }  (         0.122, 2.16291082757,  10213.28554621100 ),
  { 4312   68 }  (         0.118, 0.45789822268,   7058.59846131540 ),
  { 4312   69 }  (         0.141, 2.34932647403,  11506.76976979360 ),
  { 4312   70 }  (         0.100, 0.85621569847,   6290.18939699220 ),
  { 4312   71 }  (         0.092, 5.10587476002,   7079.37385680780 ),
  { 4312   72 }  (         0.126, 2.65428307012,  88860.05707098669 ),
  { 4312   73 }  (         0.106, 5.85646710022,   7860.41939243920 ),
  { 4312   74 }  (         0.084, 3.57457554262,  16730.46368959580 ),
  { 4312   75 }  (         0.089, 4.21433259618,  83996.84731811189 ),
  { 4312   76 }  (         0.097, 5.57938280855,  13367.97263110660 ),
  { 4312   77 }  (         0.102, 2.05853060226,     87.30820453981 ),
  { 4312   78 }  (         0.080, 4.73792651816,  11926.25441366880 ),
  { 4312   79 }  (         0.080, 5.41418965044,  10973.55568635000 ),
  { 4312   80 }  (         0.106, 4.10978997399,   3496.03282613400 ),
  { 4312   81 }  (         0.102, 3.62650006043, 244287.60000722769 ),
  { 4312   82 }  (         0.075, 4.89483161769,   5643.17856367740 ),
  { 4312   83 }  (         0.087, 0.42863750683,  11015.10647733480 ),
  { 4312   84 }  (         0.069, 1.88908760720,  10177.25767953360 ),
  { 4312   85 }  (         0.089, 1.35567273119,   6681.22485339960 ),
  { 4312   86 }  (         0.066, 0.99455837265,   6525.80445396540 ),
  { 4312   87 }  (         0.067, 5.51240997070,   3097.88382272579 ),
  { 4312   88 }  (         0.076, 2.72016814799,   4164.31198961300 ),
  { 4312   89 }  (         0.063, 1.44349902540,   9917.69687450980 ),
  { 4312   90 }  (         0.078, 3.51469733747,  11856.21865142450 ),
  { 4312   91 }  (         0.085, 0.50956043858,  10575.40668294180 ),
  { 4312   92 }  (         0.067, 3.62043033405,  16496.36139620240 ),
  { 4312   93 }  (         0.055, 5.24637517308,   3340.61242669980 ),
  { 4312   94 }  (         0.048, 5.43966777314,  20426.57109242200 ),
  { 4312   95 }  (         0.064, 5.79535817813,   2388.89402044920 ),
  { 4312   96 }  (         0.046, 5.43499966519,   6275.96230299060 ),
  { 4312   97 }  (         0.050, 3.86263598617,   5729.50644714900 ),
  { 4312   98 }  (         0.044, 1.52269529228,  12168.00269657460 ),
  { 4312   99 }  (         0.057, 4.96352373486,  14945.31617355440 ),
  { 4312  100 }  (         0.045, 1.00861230160,   8635.94200376320 ),
  { 4312  101 }  (         0.043, 3.30685683359,   9779.10867612540 ),
  { 4312  102 }  (         0.042, 0.63481258930,   2699.73481931760 ),
  { 4312  103 }  (         0.041, 5.67996766641,  11712.95531823080 ),
  { 4312  104 }  (         0.056, 4.34024451468,  90955.55169449610 ),
  { 4312  105 }  (         0.041, 5.81722212845,    709.93304855830 ),
  { 4312  106 }  (         0.053, 6.17052087143, 233141.31440436149 ),
  { 4312  107 }  (         0.037, 3.12495025087,  16200.77272450120 ),
  { 4312  108 }  (         0.035, 5.76973458495,  12569.67481833180 ),
  { 4312  109 }  (         0.037, 0.31656444326,  24356.78078864160 ),
  { 4312  110 }  (         0.035, 0.96229051027,  17298.18232732620 ),
  { 4312  111 }  (         0.033, 5.23130355867,   5331.35744374080 ),
  { 4312  112 }  (         0.035, 0.62517020593,  25158.60171976540 ),
  { 4312  113 }  (         0.035, 0.80004512129,  13916.01910964160 ),
  { 4312  114 }  (         0.037, 2.89336088688,  12721.57209941700 ),
  { 4312  115 }  (         0.030, 4.50198402401,  23543.23050468179 ),
  { 4312  116 }  (         0.030, 5.31355708693,  18319.53658487960 ),
  { 4312  117 }  (         0.029, 3.47275229977,  13119.72110282519 ),
  { 4312  118 }  (         0.029, 3.11002782516,   4136.91043351620 ),
  { 4312  119 }  (         0.032, 5.52273255667,   5753.38488489680 ),
  { 4312  120 }  (         0.035, 3.79699996680, 143571.32428481648 ),
  { 4312  121 }  (         0.026, 1.50634201907, 154717.60988768269 ),
  { 4312  122 }  (         0.030, 3.53519084118,   6284.05617105960 ),
  { 4312  123 }  (         0.023, 4.41808025967,   5884.92684658320 ),
  { 4312  124 }  (         0.025, 1.38477355808,  65147.61976813770 ),
  { 4312  125 }  (         0.023, 3.49782549797,   7477.52286021600 ),
  { 4312  126 }  (         0.019, 3.14329413716,   6496.37494542940 ),
  { 4312  127 }  (         0.019, 2.20135125199,  18073.70493865020 ),
  { 4312  128 }  (         0.019, 4.95020255309,   3930.20969621960 ),
  { 4312  129 }  (         0.019, 0.57998702747,  31415.37924995700 ),
  { 4312  130 }  (         0.021, 1.75474323399,  12139.55350910680 ),
  { 4312  131 }  (         0.019, 3.92233070499,  19651.04848109800 ),
  { 4312  132 }  (         0.014, 0.98131213224,  12559.03815298200 ),
  { 4312  133 }  (         0.019, 4.93309333729,   2942.46342329160 ),
  { 4312  134 }  (         0.016, 5.55997534558,   8827.39026987480 ),
  { 4312  135 }  (         0.013, 1.68808165516,   4535.05943692440 ),
  { 4312  136 }  (         0.013, 0.33982116161,   4933.20844033260 ),
  { 4312  137 }  (         0.012, 1.85426309994,   5856.47765911540 ),
  { 4312  138 }  (         0.010, 4.82763996845,  13095.84266507740 ),
  { 4312  139 }  (         0.011, 5.38005490571,  11790.62908865880 ),
  { 4312  140 }  (         0.010, 1.40815507226,  10988.80815753500 ),
  { 4312  141 }  (         0.011, 3.05005267431,  17260.15465469040 ),
  { 4312  142 }  (         0.010, 4.93364992366,  12352.85260454480 )
  (*$endif *)
      );
  (*$ifdef meeus *)
  vsop87_ear_l3:array[0..  6,0..2] of extended = (
  (*$else *)
  vsop87_ear_l3:array[0.. 21,0..2] of extended = (
  (*$endif *)
  { 4313    1 }  (       289.226, 5.84384198723,   6283.07584999140 ),
  { 4313    2 }  (        34.955, 0.00000000000,      0.00000000000 ),
  { 4313    3 }  (        16.819, 5.48766912348,  12566.15169998280 ),
  { 4313    4 }  (         2.962, 5.19577265202,    155.42039943420 ),
  { 4313    5 }  (         1.288, 4.72200252235,      3.52311834900 ),
  { 4313    6 }  (         0.635, 5.96925937141,    242.72860397400 ),
  { 4313    7 }  (         0.714, 5.30045809128,  18849.22754997420 )
  (*$ifndef meeus *)                                                 ,
  { 4313    8 }  (         0.402, 3.78682982419,    553.56940284240 ),
  { 4313    9 }  (         0.072, 4.29768126180,   6286.59896834040 ),
  { 4313   10 }  (         0.067, 0.90721687647,   6127.65545055720 ),
  { 4313   11 }  (         0.036, 5.24029648014,   6438.49624942560 ),
  { 4313   12 }  (         0.024, 5.16003960716,  25132.30339996560 ),
  { 4313   13 }  (         0.023, 3.01921570335,   6309.37416979120 ),
  { 4313   14 }  (         0.017, 5.82863573502,   6525.80445396540 ),
  { 4313   15 }  (         0.017, 3.67772863930,  71430.69561812909 ),
  { 4313   16 }  (         0.009, 4.58467294499,   1577.34354244780 ),
  { 4313   17 }  (         0.008, 1.40626662824,  11856.21865142450 ),
  { 4313   18 }  (         0.008, 5.07561257196,   6256.77753019160 ),
  { 4313   19 }  (         0.007, 2.82473374405,  83996.84731811189 ),
  { 4313   20 }  (         0.005, 2.71488713339,  10977.07880469900 ),
  { 4313   21 }  (         0.005, 3.76879847273,  12036.46073488820 ),
  { 4313   22 }  (         0.005, 4.28412873331,   6275.96230299060 )
  (*$endif *)
      );
  (*$ifdef meeus *)
  vsop87_ear_l4:array[0..  2,0..2] of extended = (
  (*$else *)
  vsop87_ear_l4:array[0.. 10,0..2] of extended = (
  (*$endif *)
  { 4314    1 }  (       114.084, 3.14159265359,      0.00000000000 ),
  { 4314    2 }  (         7.717, 4.13446589358,   6283.07584999140 ),
  { 4314    3 }  (         0.765, 3.83803776214,  12566.15169998280 )
  (*$ifndef meeus *)                                                 ,
  { 4314    4 }  (         0.420, 0.41925861858,    155.42039943420 ),
  { 4314    5 }  (         0.040, 3.59847585840,  18849.22754997420 ),
  { 4314    6 }  (         0.041, 3.14398414077,      3.52311834900 ),
  { 4314    7 }  (         0.035, 5.00298940826,   5573.14280143310 ),
  { 4314    8 }  (         0.013, 0.48794833701,  77713.77146812050 ),
  { 4314    9 }  (         0.010, 5.64801766350,   6127.65545055720 ),
  { 4314   10 }  (         0.008, 2.84160570605, 161000.68573767410 ),
  { 4314   11 }  (         0.002, 0.54912904658,   6438.49624942560 )
  (*$endif *)
      );
  (*$ifdef meeus *)
  vsop87_ear_l5:array[0..  0,0..2] of extended = (
  (*$else *)
  vsop87_ear_l5:array[0..  4,0..2] of extended = (
  (*$endif *)
  { 4315    1 }  (         0.878, 3.14159265359,      0.00000000000 )
  (*$ifndef meeus *)                                                 ,
  { 4315    2 }  (         0.172, 2.76579069510,   6283.07584999140 ),
  { 4315    3 }  (         0.050, 2.01353298182,    155.42039943420 ),
  { 4315    4 }  (         0.028, 2.21496423926,  12566.15169998280 ),
  { 4315    5 }  (         0.005, 1.75600058765,  18849.22754997420 )
  (*$endif *)
      );
begin
  WITH result do begin
    a:=0; b:=0; c:=0;
    case index of
      0: if (nr>=low(vsop87_ear_l0)) and (nr<=high(vsop87_ear_l0)) then begin
        a:=vsop87_ear_l0[nr,0];  b:=vsop87_ear_l0[nr,1];  c:=vsop87_ear_l0[nr,2];
        end;
      1: if (nr>=low(vsop87_ear_l1)) and (nr<=high(vsop87_ear_l1)) then begin
        a:=vsop87_ear_l1[nr,0];  b:=vsop87_ear_l1[nr,1];  c:=vsop87_ear_l1[nr,2];
        end;
      2: if (nr>=low(vsop87_ear_l2)) and (nr<=high(vsop87_ear_l2)) then begin
        a:=vsop87_ear_l2[nr,0];  b:=vsop87_ear_l2[nr,1];  c:=vsop87_ear_l2[nr,2];
        end;
      3: if (nr>=low(vsop87_ear_l3)) and (nr<=high(vsop87_ear_l3)) then begin
        a:=vsop87_ear_l3[nr,0];  b:=vsop87_ear_l3[nr,1];  c:=vsop87_ear_l3[nr,2];
        end;
      4: if (nr>=low(vsop87_ear_l4)) and (nr<=high(vsop87_ear_l4)) then begin
        a:=vsop87_ear_l4[nr,0];  b:=vsop87_ear_l4[nr,1];  c:=vsop87_ear_l4[nr,2];
        end;
      5: if (nr>=low(vsop87_ear_l5)) and (nr<=high(vsop87_ear_l5)) then begin
        a:=vsop87_ear_l5[nr,0];  b:=vsop87_ear_l5[nr,1];  c:=vsop87_ear_l5[nr,2];
        end;
      end;
    end;
  end;

initialization
  datetime_2000_01_01:=Encodedate(2000,1,1);
end.

